/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <mpcdec/streaminfo.h>
#include <mpcdec/mpcdec.h>
#include "internal.h"
#include "decoder.h"

// mpc_bits_reader.c
mpc_uint32_t mpc_bits_read(mpc_bits_reader * r, const unsigned int nb_bits);
int mpc_bits_get_block(mpc_bits_reader * r, mpc_block * p_block);

// streaminfo.c
mpc_status streaminfo_read_header_sv8(mpc_streaminfo* si, const mpc_bits_reader * r_in);
mpc_status streaminfo_read_header_sv7(mpc_streaminfo* si, mpc_bits_reader * r_in);

#define MAX_FRAME_SIZE 4352

enum {
	MPC_BUFFER_CLEAR = 1,
	MPC_BUFFER_SWAP = 2,
	MPC_BUFFER_FULL = 4,
};

static mpc_uint32_t
mpc_demux_fill(mpc_demux * d, mpc_uint32_t min_bytes, int flags)
{
	mpc_uint32_t unread_bytes = d->bytes_total + d->buffer - d->bits_reader.buff
			- ((8 - d->bits_reader.count) >> 3);

	if (min_bytes == 0 || min_bytes > DEMUX_BUFFER_SIZE ||
		    (unread_bytes < min_bytes && flags & MPC_BUFFER_FULL))
		min_bytes = DEMUX_BUFFER_SIZE;

	if (unread_bytes < min_bytes) {
		mpc_uint32_t bytes2read = min_bytes - unread_bytes;
		mpc_uint32_t bytes_free = DEMUX_BUFFER_SIZE - d->bytes_total;

		if (flags & MPC_BUFFER_SWAP)
			bytes2read &= -1 << 2;

		if (bytes2read > bytes_free) {
			if (d->bits_reader.count == 0) {
				d->bits_reader.count = 8;
				d->bits_reader.buff++;
			}
			memmove(d->buffer, d->bits_reader.buff, unread_bytes);
			d->bits_reader.buff = d->buffer;
			d->bytes_total = unread_bytes;
		}
		bytes2read = d->r->read(d->r, d->buffer + d->bytes_total, bytes2read);
		if (flags & MPC_BUFFER_SWAP){
			unsigned int i, * tmp = (unsigned int *) (d->buffer + d->bytes_total);
			for(i = 0 ;i < (bytes2read >> 2); i++)
				tmp[i] = mpc_swap32(tmp[i]);
		}
		d->bytes_total += bytes2read;
		return bytes2read;
	}

	return 0;
}

static mpc_status mpc_demux_header(mpc_demux * d)
{
	char magic[4];
	int err;

	memset(&d->si, 0, sizeof d->si);

    // get header position
	err = d->si.header_position = mpc_skip_id3v2(d->r);
	if(err < 0) return MPC_STATUS_FILE;

	// seek to first byte of mpc data
	err = d->r->seek(d->r, d->si.header_position);
	if(!err) return MPC_STATUS_FILE;

	d->si.tag_offset = d->si.total_file_length = d->r->get_size(d->r);

	mpc_demux_fill(d, 4, 0);
	magic[0] = mpc_bits_read(&d->bits_reader, 8);
	magic[1] = mpc_bits_read(&d->bits_reader, 8);
	magic[2] = mpc_bits_read(&d->bits_reader, 8);
	magic[3] = mpc_bits_read(&d->bits_reader, 8);

	if (memcmp(magic, "MP+", 3) == 0) {
		d->si.stream_version = magic[3];
		if ((d->si.stream_version & 15) == 7) {
			mpc_demux_fill(d, 6 * 4, MPC_BUFFER_SWAP); // header block size + endian convertion
			streaminfo_read_header_sv7(&d->si, &d->bits_reader);
		} else {
			return MPC_STATUS_INVALIDSV;
		}
	} else if (memcmp(magic, "MPCK", 4) == 0) {
		mpc_block b;
		int size;
		mpc_demux_fill(d, 11, 0); // max header block size
		size = mpc_bits_get_block(&d->bits_reader, &b);
		while( memcmp(b.key, "AD", 2) != 0 ){ // scan all blocks until audio
			mpc_demux_fill(d, 11 + b.size, 0);
			if (memcmp(b.key, "SI", 2) == 0)
				streaminfo_read_header_sv8(&d->si, &d->bits_reader);
			d->bits_reader.buff += b.size;
			size = mpc_bits_get_block(&d->bits_reader, &b);
		}
		d->bits_reader.buff -= size;
	} else
		return MPC_STATUS_INVALIDSV;

	return MPC_STATUS_OK;
}

mpc_demux * mpc_demux_init(mpc_reader * p_reader)
{
	mpc_demux* p_tmp = malloc(sizeof(mpc_demux));

	if (p_tmp != 0) {
		p_tmp->r = p_reader;
		p_tmp->bytes_total = 0;
		p_tmp->bits_reader.buff = p_tmp->buffer;
		p_tmp->bits_reader.count = 8;
		p_tmp->block_bits = 0;
		// lire entête
		mpc_demux_header(p_tmp);
		// initialisation decodeur
		p_tmp->d = mpc_decoder_init(&p_tmp->si);
	}

	return p_tmp;
}

void mpc_demux_exit(mpc_demux * d)
{
	mpc_decoder_exit(d->d);
	free(d);
}

void mpc_demux_get_info(mpc_demux * d, mpc_streaminfo * i)
{
	memcpy(i, &d->si, sizeof d->si);
}

void mpc_demux_decode(mpc_demux * d, mpc_frame_info * i)
{
	if (d->si.stream_version == 8) {
		mpc_bits_reader r;
		if (d->block_bits < 8 && (d->d->decoded_frames < d->d->frames || d->d->frames == 0)){
			mpc_block b;
			d->bits_reader.count &= -8;
			mpc_demux_fill(d, 11, 0); // max header block size
			mpc_bits_get_block(&d->bits_reader, &b);
			while( memcmp(b.key, "AD", 2) != 0 ){ // scan all blocks until audio
				mpc_demux_fill(d, 11 + b.size, 0);
				d->bits_reader.buff += b.size;
				mpc_bits_get_block(&d->bits_reader, &b);
			}
			d->block_bits = b.size * 8;
		}

		mpc_demux_fill(d, (d->block_bits >> 3) + 1, 0);
		r = d->bits_reader;
		mpc_decoder_decode_frame(d->d, &d->bits_reader, i);
		d->block_bits -= (d->bits_reader.buff - r.buff) * 8 + r.count - d->bits_reader.count;
	} else {
		mpc_bits_reader r;
		mpc_demux_fill(d, MAX_FRAME_SIZE, MPC_BUFFER_FULL | MPC_BUFFER_SWAP);
		mpc_bits_read(&d->bits_reader, 20); // read frame size
		r = d->bits_reader;
		mpc_decoder_decode_frame(d->d, &d->bits_reader, i);
	}
}

// static mpc_uint32_t
// mpc_decoder_decode_internal(mpc_decoder *d, MPC_SAMPLE_FORMAT *buffer)
// {
// 	mpc_uint32_t output_frame_length = MPC_FRAME_LENGTH;
// 	mpc_uint32_t FwdJumpInfo = 0;
// 	mpc_uint32_t  FrameBitCnt = 0;
//
// 	if (d->decoded_frames >= d->OverallFrames) {
// 		return (mpc_uint32_t)(-1);                           // end of file -> abort decoding
// 	}
//
//     // add seeking info
// 	if (d->seeking_table_frames < d->decoded_frames &&
// 		   (d->decoded_frames & ((1 << d->seeking_pwr) - 1)) == 0) {
// 		d->seeking_table[d->decoded_frames >> d->seeking_pwr] = mpc_decoder_bits_read(d);
// 		d->seeking_table_frames = d->decoded_frames;
// 		   }
//
// 		   switch (d->StreamVersion) {
// 			   case 0x07:
// 			   case 0x17:
// 		// read jump-info for validity check of frame
// 				   FwdJumpInfo  = mpc_decoder_bitstream_read(d, 20);
//     	// decode data and check for validity of frame
// 				   FrameBitCnt = mpc_decoder_bits_read(d);
//
// 				   d->FrameWasValid = mpc_decoder_bits_read(d) - FrameBitCnt == FwdJumpInfo;
// 				   break;
// 			   case 0x08:
// 				   if (d->BlockBits < 8){
// 					   mpc_block block;
// 			// trouver le block
// 					   mpc_decoder_bitstream_read(d, d->BlockBits);
// 					   while (1) {
// 						   mpc_decoder_get_block(d, &block);
// 						   if (memcmp(block.key, "AD", 2) != 0)
// 							   mpc_decoder_seek(d, mpc_decoder_bits_read(d) + block.size * 8);
// 						   else
// 							   break;
// 					   }
// 					   d->BlockBits = block.size * 8;
// 				   }
// 				   FrameBitCnt = mpc_decoder_bits_read(d);
// 				   mpc_decoder_read_bitstream_sv7(d, FALSE);
// 				   d->BlockBits -= mpc_decoder_bits_read(d) - FrameBitCnt;
// 				   d->FrameWasValid = 1;
// 				   break;
// 			   default:
// 				   return (mpc_uint32_t)(-1);
// 		   }
//
// 		   return output_frame_length;
// }
//
// mpc_uint32_t mpc_decoder_decode(
// 		mpc_decoder *d,
//     MPC_SAMPLE_FORMAT *buffer,
// 	mpc_uint32_t *vbr_update_acc,
// 	mpc_uint32_t *vbr_update_bits)
// {
// 	for(;;)
// 	{
//         //const mpc_int32_t MaxBrokenFrames = 0; // PluginSettings.MaxBrokenFrames
//
// 		mpc_uint32_t RING = d->Zaehler;
// 		mpc_int32_t vbr_ring = (RING << 5) + d->pos;
//
// 		mpc_uint32_t valid_samples = mpc_decoder_decode_internal(d, buffer);
//
// 		if (valid_samples == (mpc_uint32_t)(-1) ) {
// 			return 0;
// 		}
//
// 		/**************** ERROR CONCEALMENT *****************/
// 		if (d->FrameWasValid == 0 ) {
//             // error occurred in bitstream
// 			return (mpc_uint32_t)(-1);
// 		}
// 		else {
// 			if (vbr_update_acc && vbr_update_bits) {
// 				(*vbr_update_acc) ++;
// 				vbr_ring = (d->Zaehler << 5) + d->pos - vbr_ring;
// 				if (vbr_ring < 0) {
// 					vbr_ring += 524288;
// 				}
// 				(*vbr_update_bits) += vbr_ring;
// 			}
//
// 		}
// 		mpc_decoder_update_buffer(d, RING);
//
// 		if (valid_samples > 0) {
// 			return valid_samples;
// 		}
// 	}
// }

