/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>

#include "libmpcenc.h"
#include "stdio.h"

unsigned long crc32(unsigned char *buf, int len);

void emptyBits(mpc_encoder_t * e)
{
	while( e->bitsCount >= 8 ){
		e->bitsCount -= 8;
		e->buffer[e->pos] = (mpc_uint8_t) (e->bitsBuff >> e->bitsCount);
		e->pos++;
	}
}

void writeBits (mpc_encoder_t * e, mpc_uint32_t input, unsigned int bits )
{
	e->outputBits += bits;

	if (e->bitsCount + bits > sizeof(e->bitsBuff) * 8) {
		int tmp = (sizeof(e->bitsBuff) * 8 - e->bitsCount);
		bits -= tmp;
		e->bitsBuff = (e->bitsBuff << tmp) | (input >> bits);
		e->bitsCount = sizeof(e->bitsBuff) * 8;
		emptyBits(e);
		input &= (1 << bits) - 1;
	}
	e->bitsBuff = (e->bitsBuff << bits) | input;
	e->bitsCount += bits;
}

unsigned int encodeSize(mpc_uint64_t size, char * buff, mpc_bool_t addCodeSize)
{
	unsigned int i = 1;
	int j;

	if (addCodeSize) {
		while ((1 << (7 * i)) - i <= size) i++;
		size += i;
	} else
		while ((1 << (7 * i)) <= size) i++;

	for( j = i - 1; j >= 0; j--){
		buff[j] = (char) (size | 0x80);
		size >>= 7;
	}
	buff[i - 1] &= 0x7F;

	return i;
}

static void encodeGolomb(mpc_encoder_t * e, mpc_uint32_t nb, mpc_uint_t k)
{
	unsigned int l = (nb >> k) + 1;
	nb &= (1 << k) - 1;

	while( l > 32 ){
		writeBits(e, 0, 32);
		l -= 32;
	}
	writeBits(e, 1, l);
	writeBits(e, nb, k);
}

void writeMagic(mpc_encoder_t * e)
{
	fwrite("MPCK", sizeof(char), 4, e->outputFile);
	e->outputBits += 32;
	e->framesInBlock = 0;
}

void writeBlock ( mpc_encoder_t * e, const char * key, const mpc_bool_t addCRC)
{
	FILE * fp = e->outputFile;
	mpc_uint32_t written = 0;
	mpc_uint8_t * datas = e->buffer;
	char blockSize[10];
	mpc_uint_t len;

	writeBits(e, 0, (8 - e->bitsCount) % 8);
	emptyBits(e);

	// write block header (key / length)
	len = encodeSize(e->pos + 2 + (addCRC ? 4 : 0), blockSize, MPC_TRUE);
	fwrite(key, sizeof(char), 2, fp);
	fwrite(blockSize, sizeof(char), len, fp);
	e->outputBits += (len + 2) * 8;


	if (addCRC) {
		char tmp[4];
		unsigned long CRC32 = crc32((unsigned char *) e->buffer, e->pos);
		tmp[0] = (char) (CRC32 >> 24);
		tmp[1] = (char) (CRC32 >> 16);
		tmp[2] = (char) (CRC32 >> 8);
		tmp[3] = (char) CRC32;
		fwrite(tmp, sizeof(char), 4, fp);
		e->outputBits += 32;
	}

	// write datas
	while ( e->pos != 0 ) {
		written = fwrite ( datas, sizeof(*e->buffer), e->pos, fp );
		if ( written == 0 ) {
			fprintf(stderr, "\b\n WARNING: Disk full?, retry after 10 sec ...\a");
            sleep (10);
        }
		if ( written > 0 ) {
			datas += written;
			e->pos -= written;
        }
	}
	e->framesInBlock = 0;
}

void writeSeekTable (mpc_encoder_t * e)
{
	mpc_uint32_t tmp1, tmp2, i, len;
	mpc_uint32_t * table = e->seek_table;
	mpc_uint8_t tmp[10];

	// write the position to header
	tmp1 = ftell(e->outputFile); // get the seek table position
	tmp[0] = (mpc_uint8_t) (tmp1 >> 24);
	tmp[1] = (mpc_uint8_t) (tmp1 >> 16);
	tmp[2] = (mpc_uint8_t) (tmp1 >> 8);
	tmp[3] = (mpc_uint8_t) tmp1;
	fseek(e->outputFile, e->seek_ref + 3, SEEK_SET);
	fwrite(tmp, sizeof(mpc_uint8_t), 4, e->outputFile);
	fseek(e->outputFile, tmp1, SEEK_SET);

	tmp1 = table[0];
	tmp2 = table[1];

	len = encodeSize(tmp1 - e->seek_ref, tmp, MPC_FALSE);
	for( i = 0; i < len; i++)
		writeBits ( e, tmp[i], 8 );
	len = encodeSize(tmp2 - e->seek_ref, tmp, MPC_FALSE);
	for( i = 0; i < len; i++)
		writeBits ( e, tmp[i], 8 );

	for( i = 2; i < e->seek_pos; i++){
		int code = table[i] - 2 * tmp2 + tmp1;
		tmp1 = tmp2;
		tmp2 = table[i];
		if (code >= 0) {
			writeBits(e, 0, 1);
			encodeGolomb(e, code, 10);
		} else {
			writeBits(e, 1, 1);
			encodeGolomb(e, -code, 10);
		}
	}
}

/* end of bitstream.c */
