/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>

#include "libmpcenc.h"

// bitstream.c
void writeBits (mpc_encoder_t * e, mpc_uint32_t input, unsigned int bits );
unsigned int encodeSize(mpc_uint64_t, char *, mpc_bool_t);
void writeBlock ( mpc_encoder_t * e, const char * key, const mpc_bool_t addCRC);

void       Init_Huffman_Encoder_SV7   ( void );

void   Klemm ( void );

void   Init_Skalenfaktoren             ( void );

// huffsv7.c
extern Huffman_t         HuffHdr  [10];         // contains tables for SV7-header
extern Huffman_t         HuffSCFI [ 4];         // contains tables for SV7-scalefactor select
extern Huffman_t         HuffDSCF [16];         // contains tables for SV7-scalefactor coding
extern const Huffman_t*  HuffQ [2] [8];         // points to tables for SV7-sample coding

/*
 *  SV1:   DATE 13.12.1998
 *  SV2:   DATE 12.06.1999
 *  SV3:   DATE 19.10.1999
 *  SV4:   DATE 20.10.1999
 *  SV5:   DATE 18.06.2000
 *  SV6:   DATE 10.08.2000
 *  SV7:   DATE 23.08.2000
 *  SV7.f: DATE 20.07.2002
 */

unsigned char         MS_Flag     [32];         // Flag to save if Subband was MS- or LR-coded
int                   SCF_Last_L  [32];
int                   SCF_Last_R  [32];         // Last coded SCF value
static unsigned char  DSCF_RLL_L  [32];
static unsigned char  DSCF_RLL_R  [32];         // Duration of the differential SCF-coding for RLL (run length limitation)
int                   Res_L       [32];
int                   Res_R       [32];         // Quantization precision of the subbands
int                   SCF_Index_L [32] [3];
int                   SCF_Index_R [32] [3];     // Scalefactor index for quantized subband values


// initialize SV8
void
Init_SV8 ( mpc_encoder_t * e )
{
    Init_Huffman_Encoder_SV7 ();
	Init_Skalenfaktoren ();
	Klemm    ();

	e->pos = 0;
	e->bitsCount = 0;
	e->outputBits = 0;
	e->bitsBuff = 0;
	e->Overflows = 0;
	e->seek_pos = 0;
}


// writes SV8-header
void
writeStreamInfo ( mpc_encoder_t*e,
				  const unsigned int  MaxBand,
                  const unsigned int  MS_on,
                  const unsigned int  SamplesCount,
                  const unsigned int  SampleFreq,
				  const unsigned int  ChannelCount)
{
	unsigned char samplesCount[10];
	int samplesCountLen = encodeSize(SamplesCount, (char *)samplesCount, MPC_FALSE);
	int i;

    writeBits ( e, 0x08,  8 );    // StreamVersion

	for( i = 0; i < samplesCountLen; i++) // nb of samples
		writeBits ( e, samplesCount[i]  , 8 );

	switch ( SampleFreq ) {
		case 44100: writeBits ( e, 0, 4 ); break;
		case 48000: writeBits ( e, 1, 4 ); break;
		case 37800: writeBits ( e, 2, 4 ); break;
		case 32000: writeBits ( e, 3, 4 ); break;
		default   : fprintf(stderr, "Internal error\n");// FIXME : stderr_printf ( "Internal error\n");
		exit (1);
	}

	writeBits ( e, ChannelCount - 1  ,  4 );    // Channels
	writeBits ( e, MaxBand - 1  ,  5 );    // Bandwidth
	writeBits ( e, MS_on        ,  1 );    // MS-Coding Flag
	writeBits ( e, FRAMES_PER_BLOCK_PWR,  4 );    // frames per block (log2 unit)
}

// writes encoder signature
void writeEncoderInfo ( mpc_encoder_t * e,
						const int profile,
						const int PNS_on,
						const int version_major,
						const int version_minor,
						const int version_implement,
						const int version_build )
{
	writeBits ( e, profile,  4 );
	writeBits ( e, PNS_on,  1 );
	writeBits ( e, 0,  3 ); // unused
	writeBits ( e, version_major,  4 );
	writeBits ( e, version_minor,  8 );
	writeBits ( e, version_implement,  4 );
	writeBits ( e, version_build,  8 );
}

// writes replay gain info
void writeGainInfo ( mpc_encoder_t * e )
{
	writeBits ( e, 1,  8 ); // version
	writeBits ( e, 0,  16 ); // Title gain
	writeBits ( e, 0,  16 ); // Title peak
	writeBits ( e, 0,  16 ); // Album gain
	writeBits ( e, 0,  16 ); // Album peak
}

#define ENCODE_SCF1( new, old, rll )                         \
        d = new - old + 7;                                   \
        if ( d <= 14u  && rll < 1) {                        \
            writeBits ( e, Table[d].Code, Table[d].Length );    \
        } else {                                               \
            if ( new < 0 ) new = 0, e->Overflows++;          \
            writeBits ( e, Table[15].Code, Table[15].Length );  \
            writeBits ( e, (unsigned int)new, 6 );              \
            rll = 0;                                         \
        }

#define ENCODE_SCFn( new, old, rll )                         \
        d = new - old + 7;                                   \
        if ( d <= 14u ) {                                    \
            writeBits ( e, Table[d].Code, Table[d].Length );    \
        } else {                                               \
            if ( new < 0 ) new = 0, e->Overflows++;          \
            writeBits ( e, Table[15].Code, Table[15].Length );  \
            writeBits ( e, (unsigned int)new, 6 );              \
            rll = 0;                                         \
        }


// formatting and writing SV7-bitstream for one frame
void
WriteBitstream_SV7 ( mpc_encoder_t* e,
					 const int               MaxBand,
                     const SubbandQuantTyp*  Q )
{
    int                  n;
    int                  k;
    unsigned int         d;
    unsigned int         idx;
    unsigned int         book;
    const Huffman_t*     Table;
    const Huffman_t*     Table0;
    const Huffman_t*     Table1;
    int                  sum;
    const unsigned int*  q;
    unsigned char        SCFI_L [32];
    unsigned char        SCFI_R [32];

    /************************************ Resolution *********************************/
    writeBits ( e, (unsigned int)Res_L[0], 4 );                            // subband 0
	writeBits ( e, (unsigned int)Res_R[0], 4 );
    if ( e->MS_Channelmode > 0  &&  !(Res_L[0]==0  &&  Res_R[0]==0) )
		writeBits ( e, MS_Flag[0] , 1 );

    Table = HuffHdr;                                                    // subband 1...MaxBand
    for ( n = 1; n <= MaxBand; n++ ) {
        d = Res_L[n] - Res_L[n-1] + 5;
        if ( d <= 8u ) {
			writeBits ( e, Table[d].Code, Table[d].Length );
        }
        else {
			writeBits ( e, Table[9].Code, Table[9].Length );
			writeBits ( e, Res_L[n]     , 4               );
        }

        d = Res_R[n] - Res_R[n-1] + 5;
        if ( d <= 8u ) {
			writeBits ( e, Table[d].Code, Table[d].Length );
        }
        else {
			writeBits ( e, Table[9].Code, Table[9].Length );
			writeBits ( e, Res_R[n]     , 4               );
        }
        if ( e->MS_Channelmode > 0  &&  !(Res_L[n]==0 && Res_R[n]==0) )
			writeBits ( e, MS_Flag[n], 1 );
    }

    /************************************ SCF encoding type ***********************************/
    Table = HuffSCFI;
    for ( n = 0; n <= MaxBand; n++ ) {
        if ( Res_L[n] ) {
            SCFI_L[n] = 2 * (SCF_Index_L[n][0] == SCF_Index_L[n][1]) + (SCF_Index_L[n][1] == SCF_Index_L[n][2]);
			writeBits ( e, Table[SCFI_L[n]].Code, Table[SCFI_L[n]].Length );
        }
        if ( Res_R[n] ) {
            SCFI_R[n] = 2 * (SCF_Index_R[n][0] == SCF_Index_R[n][1]) + (SCF_Index_R[n][1] == SCF_Index_R[n][2]);
			writeBits ( e, Table[SCFI_R[n]].Code, Table[SCFI_R[n]].Length );
        }
    }

    /************************************* SCF **********************************/
    Table = HuffDSCF;
	for ( n = 0; n <= MaxBand; n++ ) {
		if (e->framesInBlock == 0){
			DSCF_RLL_L[n] = DSCF_RLL_R[n] = 1; // new block -> force key frame
		}

        if ( Res_L[n] ) {
            switch ( SCFI_L[n] ) {
            default:
                ENCODE_SCF1 ( SCF_Index_L[n][0], SCF_Last_L [n]   , DSCF_RLL_L[n] );
                ENCODE_SCFn ( SCF_Index_L[n][1], SCF_Index_L[n][0], DSCF_RLL_L[n] );
                ENCODE_SCFn ( SCF_Index_L[n][2], SCF_Index_L[n][1], DSCF_RLL_L[n] );
                SCF_Last_L[n] = SCF_Index_L[n][2];
                break;
            case 1:
                ENCODE_SCF1 ( SCF_Index_L[n][0], SCF_Last_L [n]   , DSCF_RLL_L[n] );
                ENCODE_SCFn ( SCF_Index_L[n][1], SCF_Index_L[n][0], DSCF_RLL_L[n] );
                SCF_Last_L[n] = SCF_Index_L[n][1];
                break;
            case 2:
                ENCODE_SCF1 ( SCF_Index_L[n][0], SCF_Last_L[n]    , DSCF_RLL_L[n] );
                ENCODE_SCFn ( SCF_Index_L[n][2], SCF_Index_L[n][0], DSCF_RLL_L[n] );
                SCF_Last_L[n] = SCF_Index_L[n][2];
                break;
            case 3:
                ENCODE_SCF1 ( SCF_Index_L[n][0], SCF_Last_L[n]    , DSCF_RLL_L[n] );
                SCF_Last_L[n] = SCF_Index_L[n][0];
                break;
            }
        }

        if ( Res_R[n] ) {
            switch ( SCFI_R[n] ) {
            default:
                ENCODE_SCF1 ( SCF_Index_R[n][0], SCF_Last_R[n]    , DSCF_RLL_R[n] );
                ENCODE_SCFn ( SCF_Index_R[n][1], SCF_Index_R[n][0], DSCF_RLL_R[n] );
                ENCODE_SCFn ( SCF_Index_R[n][2], SCF_Index_R[n][1], DSCF_RLL_R[n] );
                SCF_Last_R[n] = SCF_Index_R[n][2];
                break;
            case 1:
                ENCODE_SCF1 ( SCF_Index_R[n][0], SCF_Last_R[n]    , DSCF_RLL_R[n] );
                ENCODE_SCFn ( SCF_Index_R[n][1], SCF_Index_R[n][0], DSCF_RLL_R[n] );
                SCF_Last_R[n] = SCF_Index_R[n][1];
                break;
            case 2:
                ENCODE_SCF1 ( SCF_Index_R[n][0], SCF_Last_R[n]    , DSCF_RLL_R[n] );
                ENCODE_SCFn ( SCF_Index_R[n][2], SCF_Index_R[n][0], DSCF_RLL_R[n] );
                SCF_Last_R[n] = SCF_Index_R[n][2];
                break;
            case 3:
                ENCODE_SCF1 ( SCF_Index_R[n][0], SCF_Last_R[n]    , DSCF_RLL_R[n] );
                SCF_Last_R[n] = SCF_Index_R[n][0];
                break;
            }
        }
    }

    /*********************************** Samples *********************************/
    for ( n = 0; n <= MaxBand; n++ ) {

        sum = 0;
        q   = Q[n].L;

        switch ( Res_L[n] ) {
        case -1:
        case  0:
            break;
        case  1:
            Table0 = HuffQ [0][1];
            Table1 = HuffQ [1][1];
            for ( k = 0; k < 36; k += 3 ) {
                idx  = q[k+0] + 3*q[k+1] + 9*q[k+2];
                sum += Table0 [idx].Length;
                sum -= Table1 [idx].Length;
            }
            book = sum >= 0;
			writeBits ( e, book, 1 );
            Table = HuffQ [book][1];
            for ( k = 0; k < 36; k += 3 ) {
                idx = q[k+0] + 3*q[k+1] + 9*q[k+2];
				writeBits ( e, Table[idx].Code, Table[idx].Length );
            }
            break;
        case  2:
            Table0 = HuffQ [0][2];
            Table1 = HuffQ [1][2];
            for ( k = 0; k < 36; k += 2 ) {
                idx  = q[k+0] + 5*q[k+1];
                sum += Table0 [idx].Length;
                sum -= Table1 [idx].Length;
            }
            book = sum >= 0;
			writeBits ( e, book, 1 );
            Table = HuffQ [book][2];
            for ( k = 0; k < 36; k += 2 ) {
                idx = q[k+0] + 5*q[k+1];
				writeBits ( e, Table[idx].Code, Table[idx].Length );
            }
            break;
        case  3:
        case  4:
        case  5:
        case  6:
        case  7:
            Table0 = HuffQ [0][Res_L[n]];
            Table1 = HuffQ [1][Res_L[n]];
            for ( k = 0; k < 36; k++ ) {
                sum += Table0 [q[k]].Length;
                sum -= Table1 [q[k]].Length;
            }
            book = sum >= 0;
			writeBits ( e, book, 1 );
            Table = HuffQ [book][Res_L[n]];
            for ( k = 0; k < 36; k++ ) {
                idx = q[k];
				writeBits ( e, Table[idx].Code, Table[idx].Length );
            }
            break;
        default:
            for ( k = 0; k < 36; k++ )
				writeBits ( e, q[k], Res_L[n]-1 );
            break;
        }

        sum = 0;
        q   = Q[n].R;

        switch ( Res_R[n] ) {
        case -1:
        case  0:
            break;
        case  1:
            Table0 = HuffQ [0][1];
            Table1 = HuffQ [1][1];
            for ( k = 0; k < 36; k += 3 ) {
                idx  = q[k+0] + 3*q[k+1] + 9*q[k+2];
                sum += Table0 [idx].Length;
                sum -= Table1 [idx].Length;
            }
            book = sum >= 0;
			writeBits ( e, book, 1 );
            Table = HuffQ [book][1];
            for ( k = 0; k < 36; k += 3 ) {
                idx = q[k+0] + 3*q[k+1] + 9*q[k+2];
				writeBits ( e, Table[idx].Code, Table[idx].Length );
            }
            break;
        case  2:
            Table0 = HuffQ [0][2];
            Table1 = HuffQ [1][2];
            for ( k = 0; k < 36; k += 2 ) {
                idx  = q[k+0] + 5*q[k+1];
                sum += Table0 [idx].Length;
                sum -= Table1 [idx].Length;
            }
            book = sum >= 0;
			writeBits ( e, book, 1 );
            Table = HuffQ [book][2];
            for ( k = 0; k < 36; k += 2 ) {
                idx = q[k+0] + 5*q[k+1];
				writeBits ( e, Table[idx].Code, Table[idx].Length );
            }
            break;
        case  3:
        case  4:
        case  5:
        case  6:
        case  7:
            Table0 = HuffQ [0][Res_R[n]];
            Table1 = HuffQ [1][Res_R[n]];
            for ( k = 0; k < 36; k++ ) {
                sum += Table0 [q[k]].Length;
                sum -= Table1 [q[k]].Length;
            }
            book = sum >= 0;
			writeBits ( e, book, 1 );
            Table = HuffQ [book][Res_R[n]];
            for ( k = 0; k < 36; k++ ) {
                idx = q[k];
				writeBits ( e, Table[idx].Code, Table[idx].Length );
            }
            break;
        default:
            for ( k = 0; k < 36; k++ )
				writeBits ( e, q[k], Res_R[n] - 1 );
            break;
        }

    }

	e->framesInBlock++;
	if (e->framesInBlock == FRAMES_PER_BLOCK) {
		e->seek_table[e->seek_pos] = ftell(e->outputFile);
		e->seek_pos++;
		writeBlock(e, "AD", MPC_FALSE);
	}
}

#undef ENCODE_SCF1
#undef ENCODE_SCFn


#if 0
void
Dump ( const unsigned int* q, const int Res )
{
    switch ( Res ) {
    case  1:
        for ( k = 0; k < 36; k++, q++ )
            printf ("%2d%c", *q-1, k==35?'\n':' ');
        break;
    case  2:
        for ( k = 0; k < 36; k++, q++ )
            printf ("%2d%c", *q-2, k==35?'\n':' ');
        break;
    case  3: case  4: case  5: case  6: case  7:
        if ( Res == 5 )
            for ( k = 0; k < 36; k++, q++ )
                printf ("%2d%c", *q-7, k==35?'\n':' ');
        break;
    case  8: case  9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
        printf ("%2u: ", Res-1 );
        for ( k = 0; k < 36; k++, q++ ) {
            printf ("%6d", *q - (1 << (Res-2)) );
        }
        printf ("\n");
        break;
    }
}
#endif

/* end of encode_sv7.c */
