/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/// \file mpc_decoder.c
/// Core decoding routines and logic.
#include <string.h>
#include <mpcdec/mpcdec.h>
#include "decoder.h"
#include "huffman.h"
#include "internal.h"
#include "math.h"
#include "requant.h"

mpc_uint32_t mpc_bits_read(mpc_bits_reader * r, const unsigned int nb_bits);
mpc_int32_t mpc_bits_huff_dec(mpc_bits_reader * r, const mpc_huffman *Table);

//SV7 tables
extern const mpc_huffman*   mpc_table_HuffQ [2] [8];
extern const mpc_huffman    mpc_table_HuffHdr  [10];
extern const mpc_huffman    mpc_table_HuffSCFI [ 4];
extern const mpc_huffman    mpc_table_HuffDSCF [16];

//------------------------------------------------------------------------------
// types
//------------------------------------------------------------------------------
enum
{
    EQ_TAP    = 13,                  // length of FIR filter for EQ
    DELAY     = ((EQ_TAP + 1) / 2),  // delay of FIR
    FIR_BANDS = 4,                   // number of subbands to be FIR filtered
    MEMSIZE   = MPC_DECODER_MEMSIZE, // overall buffer size
    MEMSIZE2  = (MEMSIZE/2),         // size of one buffer
    MEMMASK   = (MEMSIZE-1)
};

//------------------------------------------------------------------------------
// forward declarations
//------------------------------------------------------------------------------
void mpc_decoder_read_bitstream_sv7(mpc_decoder * d, mpc_bits_reader * r);
static void mpc_decoder_requantisierung(mpc_decoder *d);

// static void mpc_decoder_reset_y(mpc_decoder *d)
// {
// 	memset(d->Y_L, 0, sizeof d->Y_L);
// 	memset(d->Y_R, 0, sizeof d->Y_R);
// }

/**
 * set the scf indexes for seeking use
 * needed only for sv7 seeking
 * @param d
 */
void mpc_decoder_reset_scf(mpc_decoder * d)
{
	memset(d->SCF_Index_L, 1, sizeof d->SCF_Index_L );
	memset(d->SCF_Index_R, 1, sizeof d->SCF_Index_R );
}


void mpc_decoder_setup(mpc_decoder *d)
{
	memset(d, 0, sizeof *d);

	d->__r1 = 1;
	d->__r2 = 1;
	d->samples_to_skip = MPC_DECODER_SYNTH_DELAY;

	mpc_decoder_init_quant(d, 1.0f);
}

void mpc_decoder_set_streaminfo(mpc_decoder *d, mpc_streaminfo *si)
{
	d->stream_version     = si->stream_version;
	d->ms                 = si->ms;
	d->max_band           = si->max_band;
	d->channels           = si->channels;

	if (si->stream_version == 7 && si->is_true_gapless)
		d->samples = ((si->samples + MPC_FRAME_LENGTH - 1) / MPC_FRAME_LENGTH) * MPC_FRAME_LENGTH;
	else
		d->samples = si->samples;
}

mpc_decoder * mpc_decoder_init(mpc_streaminfo *si)
{
	mpc_decoder* p_tmp = malloc(sizeof(mpc_decoder));

	if (p_tmp != 0) {
		mpc_decoder_setup(p_tmp);
		mpc_decoder_set_streaminfo(p_tmp, si);
	}

	return p_tmp;
}

void mpc_decoder_exit(mpc_decoder *d)
{
	free(d);
}

void mpc_decoder_decode_frame(mpc_decoder * d,
							  mpc_bits_reader * r,
							  mpc_frame_info * i)
{
	mpc_bits_reader r_sav = *r;
	mpc_int64_t samples_left;

	if (d->decoded_samples >= d->samples && d->samples != 0) {
		i->samples = 0;
		i->bits = -1;
		return;
	}

	mpc_decoder_read_bitstream_sv7(d, r);
	if (d->samples_to_skip < MPC_FRAME_LENGTH + MPC_DECODER_SYNTH_DELAY) {
		mpc_decoder_requantisierung(d);
		mpc_decoder_synthese_filter_float(d, i->buffer);
	}

	d->decoded_samples += MPC_FRAME_LENGTH;

    // reconstruct exact filelength
	if (d->decoded_samples == d->samples && d->stream_version == 7) {
		int last_frame_samples = mpc_bits_read(r, 11);
		if (last_frame_samples == 0) last_frame_samples = MPC_FRAME_LENGTH;
		d->samples += last_frame_samples - MPC_FRAME_LENGTH;
	}

	samples_left = d->samples - d->decoded_samples + MPC_DECODER_SYNTH_DELAY;
	i->samples = samples_left > MPC_FRAME_LENGTH ? MPC_FRAME_LENGTH : samples_left < 0 ? 0 : (mpc_uint32_t) samples_left;
	i->bits = (mpc_uint32_t) (((r->buff - r_sav.buff) << 3) + r_sav.count - r->count);

	if (d->samples_to_skip) {
		if (i->samples < d->samples_to_skip) {
			d->samples_to_skip -= i->samples;
			i->samples = 0;
		} else {
			i->samples -= d->samples_to_skip;
			memmove(i->buffer, i->buffer + d->samples_to_skip * d->channels,
					i->samples * d->channels * sizeof (MPC_SAMPLE_FORMAT));
			d->samples_to_skip = 0;
		}
	}
}

void
mpc_decoder_requantisierung(mpc_decoder *d)
{
    mpc_int32_t     Band;
    mpc_int32_t     n;
    MPC_SAMPLE_FORMAT facL;
    MPC_SAMPLE_FORMAT facR;
    MPC_SAMPLE_FORMAT templ;
    MPC_SAMPLE_FORMAT tempr;
    MPC_SAMPLE_FORMAT* YL;
    MPC_SAMPLE_FORMAT* YR;
    mpc_int32_t*    L;
    mpc_int32_t*    R;
	const mpc_int32_t Last_Band = d->max_band;

#ifdef MPC_FIXED_POINT
#if MPC_FIXED_POINT_FRACTPART == 14
#define MPC_MULTIPLY_SCF(CcVal, SCF_idx) \
    MPC_MULTIPLY_EX(CcVal, d->SCF[SCF_idx], d->SCF_shift[SCF_idx])
#else

#error FIXME, Cc table is in 18.14 format

#endif
#else
#define MPC_MULTIPLY_SCF(CcVal, SCF_idx) \
    MPC_MULTIPLY(CcVal, d->SCF[SCF_idx])
#endif
    // requantization and scaling of subband-samples
    for ( Band = 0; Band <= Last_Band; Band++ ) {   // setting pointers
        YL = d->Y_L[0] + Band;
        YR = d->Y_R[0] + Band;
        L  = d->Q[Band].L;
        R  = d->Q[Band].R;
        /************************** MS-coded **************************/
        if ( d->MS_Flag [Band] ) {
            if ( d->Res_L [Band] ) {
                if ( d->Res_R [Band] ) {    // M!=0, S!=0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][0]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL   = (templ = MPC_MULTIPLY_FLOAT_INT(facL,*L++))+(tempr = MPC_MULTIPLY_FLOAT_INT(facR,*R++));
                        *YR   = templ - tempr;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][1]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL   = (templ = MPC_MULTIPLY_FLOAT_INT(facL,*L++))+(tempr = MPC_MULTIPLY_FLOAT_INT(facR,*R++));
                        *YR   = templ - tempr;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][2]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL   = (templ = MPC_MULTIPLY_FLOAT_INT(facL,*L++))+(tempr = MPC_MULTIPLY_FLOAT_INT(facR,*R++));
                        *YR   = templ - tempr;
                    }
                } else {    // M!=0, S==0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                    }
                }
            } else {
                if (d->Res_R[Band])    // M==0, S!=0
                {
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YR = - (*YL = MPC_MULTIPLY_FLOAT_INT(facR,*(R++)));
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YR = - (*YL = MPC_MULTIPLY_FLOAT_INT(facR,*(R++)));
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = - (*YL = MPC_MULTIPLY_FLOAT_INT(facR,*(R++)));
                    }
                } else {    // M==0, S==0
                    for ( n = 0; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = 0;
                    }
                }
            }
        }
        /************************** LR-coded **************************/
        else {
            if ( d->Res_L [Band] ) {
                if ( d->Res_R [Band] ) {    // L!=0, R!=0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][0]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][0]);
                    for (n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][1]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][1]);
                    for (; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][2]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][2]);
                    for (; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                } else {     // L!=0, R==0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = 0;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = 0;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , (mpc_uint8_t) d->SCF_Index_L[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = 0;
                    }
                }
            }
            else {
                if ( d->Res_R [Band] ) {    // L==0, R!=0
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL = 0;
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL = 0;
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , (mpc_uint8_t) d->SCF_Index_R[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL = 0;
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                } else {    // L==0, R==0
                    for ( n = 0; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = 0;
                    }
                }
            }
        }
    }
}

void mpc_decoder_read_bitstream_sv7(mpc_decoder * d, mpc_bits_reader * r)
{
    // these arrays hold decoding results for bundled quantizers (3- and 5-step)
    static const mpc_int32_t idx30[] = { -1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1};
    static const mpc_int32_t idx31[] = { -1,-1,-1, 0, 0, 0, 1, 1, 1,-1,-1,-1, 0, 0, 0, 1, 1, 1,-1,-1,-1, 0, 0, 0, 1, 1, 1};
    static const mpc_int32_t idx32[] = { -1,-1,-1,-1,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1};
    static const mpc_int32_t idx50[] = { -2,-1, 0, 1, 2,-2,-1, 0, 1, 2,-2,-1, 0, 1, 2,-2,-1, 0, 1, 2,-2,-1, 0, 1, 2};
    static const mpc_int32_t idx51[] = { -2,-2,-2,-2,-2,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2};

    mpc_int32_t n,k;
    mpc_int32_t Max_used_Band=0;
    const mpc_huffman *Table;
    mpc_int32_t idx;
    mpc_int32_t *L   ,*R;
    mpc_int32_t *ResL,*ResR;
    mpc_uint32_t tmp;

    /***************************** Header *****************************/
    ResL  = d->Res_L;
    ResR  = d->Res_R;

    // first subband
    *ResL = mpc_bits_read(r, 4);
    *ResR = mpc_bits_read(r, 4);
	if (d->ms && !(*ResL==0 && *ResR==0)) {
        d->MS_Flag[0] = mpc_bits_read(r, 1);
    }

    // consecutive subbands
    ++ResL; ++ResR; // increase pointers
	for (n=1; n <= d->max_band; ++n, ++ResL, ++ResR)
    {
		idx   = mpc_bits_huff_dec(r, mpc_table_HuffHdr);
        *ResL = (idx!=4) ? *(ResL-1) + idx : (int) mpc_bits_read(r, 4);

		idx   = mpc_bits_huff_dec(r, mpc_table_HuffHdr);
        *ResR = (idx!=4) ? *(ResR-1) + idx : (int) mpc_bits_read(r, 4);

		if (d->ms && !(*ResL==0 && *ResR==0)) {
            d->MS_Flag[n] = mpc_bits_read(r, 1);
        }

        // only perform following procedures up to the maximum non-zero subband
        if (*ResL!=0 || *ResR!=0) {
            Max_used_Band = n;
        }
    }
    /****************************** SCFI ******************************/
    L     = d->SCFI_L;
    R     = d->SCFI_R;
    ResL  = d->Res_L;
    ResR  = d->Res_R;
    for (n=0; n <= Max_used_Band; ++n, ++L, ++R, ++ResL, ++ResR) {
		if (*ResL) *L = mpc_bits_huff_dec(r, mpc_table_HuffSCFI);
		if (*ResR) *R = mpc_bits_huff_dec(r, mpc_table_HuffSCFI);
    }

    /**************************** SCF/DSCF ****************************/
    ResL  = d->Res_L;
    ResR  = d->Res_R;
    L     = d->SCF_Index_L[0];
    R     = d->SCF_Index_R[0];
    for (n=0; n<=Max_used_Band; ++n, ++ResL, ++ResR, L+=3, R+=3) {
        if (*ResL)
        {
            switch (d->SCFI_L[n])
            {
            case 1:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[0] = (idx!=8) ? L[2] + idx : (int) mpc_bits_read(r, 6);
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[1] = (idx!=8) ? L[0] + idx : (int) mpc_bits_read(r, 6);
                L[2] = L[1];
                break;
            case 3:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[0] = (idx!=8) ? L[2] + idx : (int) mpc_bits_read(r, 6);
                L[1] = L[0];
                L[2] = L[1];
                break;
            case 2:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[0] = (idx!=8) ? L[2] + idx : (int) mpc_bits_read(r, 6);
                L[1] = L[0];
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[2] = (idx!=8) ? L[1] + idx : (int) mpc_bits_read(r, 6);
                break;
            case 0:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[0] = (idx!=8) ? L[2] + idx : (int) mpc_bits_read(r, 6);
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[1] = (idx!=8) ? L[0] + idx : (int) mpc_bits_read(r, 6);
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                L[2] = (idx!=8) ? L[1] + idx : (int) mpc_bits_read(r, 6);
                break;
            default:
                return;
            }
            if (L[0] > 1024)
                L[0] = 0x8080;
            if (L[1] > 1024)
                L[1] = 0x8080;
            if (L[2] > 1024)
                L[2] = 0x8080;
        }
        if (*ResR)
        {
            switch (d->SCFI_R[n])
            {
            case 1:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[0] = (idx!=8) ? R[2] + idx : (int) mpc_bits_read(r, 6);
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[1] = (idx!=8) ? R[0] + idx : (int) mpc_bits_read(r, 6);
                R[2] = R[1];
                break;
            case 3:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[0] = (idx!=8) ? R[2] + idx : (int) mpc_bits_read(r, 6);
                R[1] = R[0];
                R[2] = R[1];
                break;
            case 2:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[0] = (idx!=8) ? R[2] + idx : (int) mpc_bits_read(r, 6);
                R[1] = R[0];
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[2] = (idx!=8) ? R[1] + idx : (int) mpc_bits_read(r, 6);
                break;
            case 0:
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[0] = (idx!=8) ? R[2] + idx : (int) mpc_bits_read(r, 6);
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[1] = (idx!=8) ? R[0] + idx : (int) mpc_bits_read(r, 6);
                idx  = mpc_bits_huff_dec(r, mpc_table_HuffDSCF);
                R[2] = (idx!=8) ? R[1] + idx : (int) mpc_bits_read(r, 6);
                break;
            default:
                return;
            }
            if (R[0] > 1024)
                R[0] = 0x8080;
            if (R[1] > 1024)
                R[1] = 0x8080;
            if (R[2] > 1024)
                R[2] = 0x8080;
        }
    }

//     if (d->seeking == TRUE)
//         return;

    /***************************** Samples ****************************/
    ResL = d->Res_L;
    ResR = d->Res_R;
    L    = d->Q[0].L;
    R    = d->Q[0].R;
    for (n=0; n <= Max_used_Band; ++n, ++ResL, ++ResR, L+=36, R+=36)
    {
        /************** links **************/
        switch (*ResL)
        {
        case  -2: case  -3: case  -4: case  -5: case  -6: case  -7: case  -8: case  -9:
        case -10: case -11: case -12: case -13: case -14: case -15: case -16: case -17:
            L += 36;
            break;
        case -1:
            for (k=0; k<36; k++ ) {
                tmp  = mpc_random_int(d);
                *L++ = ((tmp >> 24) & 0xFF) + ((tmp >> 16) & 0xFF) + ((tmp >>  8) & 0xFF) + ((tmp >>  0) & 0xFF) - 510;
            }
            break;
        case 0:
            L += 36;// increase pointer
            break;
        case 1:
            Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][1];
            for (k=0; k<12; ++k)
            {
                idx = mpc_bits_huff_dec(r, Table);
                *L++ = idx30[idx];
                *L++ = idx31[idx];
                *L++ = idx32[idx];
            }
            break;
        case 2:
            Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][2];
            for (k=0; k<18; ++k)
            {
                idx = mpc_bits_huff_dec(r, Table);
                *L++ = idx50[idx];
                *L++ = idx51[idx];
            }
            break;
        case 3:
        case 4:
            Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][*ResL];
            for (k=0; k<36; ++k)
                *L++ = mpc_bits_huff_dec(r, Table);
            break;
        case 5:
            Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][*ResL];
            for (k=0; k<36; ++k)
                *L++ = mpc_bits_huff_dec(r, Table);
            break;
        case 6:
        case 7:
            Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][*ResL];
            for (k=0; k<36; ++k)
                *L++ = mpc_bits_huff_dec(r, Table);
            break;
        case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
            tmp = Dc[*ResL];
            for (k=0; k<36; ++k)
                *L++ = (mpc_int32_t)mpc_bits_read(r, Res_bit[*ResL]) - tmp;
            break;
        default:
            return;
        }
        /************** rechts **************/
        switch (*ResR)
        {
        case  -2: case  -3: case  -4: case  -5: case  -6: case  -7: case  -8: case  -9:
        case -10: case -11: case -12: case -13: case -14: case -15: case -16: case -17:
            R += 36;
            break;
        case -1:
                for (k=0; k<36; k++ ) {
                    tmp  = mpc_random_int(d);
                    *R++ = ((tmp >> 24) & 0xFF) + ((tmp >> 16) & 0xFF) + ((tmp >>  8) & 0xFF) + ((tmp >>  0) & 0xFF) - 510;
                }
                break;
            case 0:
                R += 36;// increase pointer
                break;
            case 1:
                Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][1];
                for (k=0; k<12; ++k)
                {
                    idx = mpc_bits_huff_dec(r, Table);
                    *R++ = idx30[idx];
                    *R++ = idx31[idx];
                    *R++ = idx32[idx];
                }
                break;
            case 2:
                Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][2];
                for (k=0; k<18; ++k)
                {
                    idx = mpc_bits_huff_dec(r, Table);
                    *R++ = idx50[idx];
                    *R++ = idx51[idx];
                }
                break;
            case 3:
            case 4:
                Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][*ResR];
                for (k=0; k<36; ++k)
                    *R++ = mpc_bits_huff_dec(r, Table);
                break;
            case 5:
                Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][*ResR];
                for (k=0; k<36; ++k)
                    *R++ = mpc_bits_huff_dec(r, Table);
                break;
            case 6:
            case 7:
                Table = mpc_table_HuffQ[mpc_bits_read(r, 1)][*ResR];
                for (k=0; k<36; ++k)
                    *R++ = mpc_bits_huff_dec(r, Table);
                break;
            case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
                tmp = Dc[*ResR];
                for (k=0; k<36; ++k)
                    *R++ = (mpc_int32_t)mpc_bits_read(r, Res_bit[*ResR]) - tmp;
                break;
            default:
                return;
        }
    }
}



