/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdlib.h>
#include <stdio.h>

#include "libmpcenc.h"
#include <mpc/minimax.h>

// bitstream.c
void writeBits (mpc_encoder_t * e, mpc_uint32_t input, unsigned int bits );
unsigned int encodeSize(mpc_uint64_t, char *, mpc_bool_t);
void encodeEnum(mpc_encoder_t * e, const mpc_uint32_t bits, const mpc_uint_t N);
void encodeLog(mpc_encoder_t * e, mpc_uint32_t value, mpc_uint32_t max);
void writeBlock ( mpc_encoder_t * e, const char * key, const mpc_bool_t addCRC);

void       Init_Huffman_Encoder_SV7   ( void );

void   Klemm ( void );

void   Init_Skalenfaktoren             ( void );

// huffsv7.c
extern const Huffman_t HuffBands [33];
extern const Huffman_t HuffRes [2][17];
extern Huffman_t HuffSCFI_1 [4];         // contains tables for SV7-scalefactor select
extern Huffman_t HuffSCFI_2 [16];         // contains tables for SV7-scalefactor select
extern Huffman_t HuffDSCF_1 [64];         // contains tables for SV7-scalefactor coding
extern Huffman_t HuffDSCF_2 [65];         // contains tables for SV7-scalefactor coding
extern const Huffman_t * HuffQ [2][8];         // points to tables for SV7-sample coding
extern const Huffman_t HuffQ9up [256];

/*
 *  SV1:   DATE 13.12.1998
 *  SV2:   DATE 12.06.1999
 *  SV3:   DATE 19.10.1999
 *  SV4:   DATE 20.10.1999
 *  SV5:   DATE 18.06.2000
 *  SV6:   DATE 10.08.2000
 *  SV7:   DATE 23.08.2000
 *  SV7.f: DATE 20.07.2002
 */

unsigned char         MS_Flag     [32];         // Flag to save if Subband was MS- or LR-coded
int                   SCF_Last_L  [32];
int                   SCF_Last_R  [32];         // Last coded SCF value
static unsigned char  DSCF_RLL_L  [32];
static unsigned char  DSCF_RLL_R  [32];         // Duration of the differential SCF-coding for RLL (run length limitation)
int                   Res_L       [32];
int                   Res_R       [32];         // Quantization precision of the subbands
int                   SCF_Index_L [32] [3];
int                   SCF_Index_R [32] [3];     // Scalefactor index for quantized subband values

// initialize SV8
void
Init_SV8 ( mpc_encoder_t * e )
{
	Init_Skalenfaktoren ();
	Klemm    ();

	memset(e, 0, sizeof(*e));
	e->seek_pwr = 1;
}

// writes replay gain info
static void writeGainInfo ( mpc_encoder_t * e )
{
	writeBits ( e, 1,  8 ); // version
	writeBits ( e, 0,  16 ); // Title gain
	writeBits ( e, 0,  16 ); // Title peak
	writeBits ( e, 0,  16 ); // Album gain
	writeBits ( e, 0,  16 ); // Album peak
}

// writes SV8-header
void
writeStreamInfo ( mpc_encoder_t*e,
				  const unsigned int  MaxBand,
                  const unsigned int  MS_on,
                  const unsigned int  SamplesCount,
                  const unsigned int  SampleFreq,
				  const unsigned int  ChannelCount)
{
	unsigned char tmp[10];
	int i, len = encodeSize(SamplesCount, (char *)tmp, MPC_FALSE);

	len = encodeSize(len + 3, (char *)tmp, MPC_TRUE);
	writeBits ( e, tmp[0]  , 8 );

    writeBits ( e, 8,  8 );    // StreamVersion

	len = encodeSize(SamplesCount, (char *)tmp, MPC_FALSE);
	for( i = 0; i < len; i++) // nb of samples
		writeBits ( e, tmp[i]  , 8 );

	switch ( SampleFreq ) {
		case 44100: writeBits ( e, 0, 3 ); break;
		case 48000: writeBits ( e, 1, 3 ); break;
		case 37800: writeBits ( e, 2, 3 ); break;
		case 32000: writeBits ( e, 3, 3 ); break;
		default   : fprintf(stderr, "Internal error\n");// FIXME : stderr_printf ( "Internal error\n");
		exit (1);
	}

	writeBits ( e, ChannelCount - 1  ,  4 );    // Channels
	writeBits ( e, MaxBand - 1  ,  5 );    // Bandwidth
	writeBits ( e, MS_on        ,  1 );    // MS-Coding Flag
	writeBits ( e, FRAMES_PER_BLOCK_PWR >> 1,  3 );    // frames per block (log4 unit)

	writeGainInfo(e);
}

// writes encoder signature
void writeEncoderInfo ( mpc_encoder_t * e,
						const int profile,
						const int PNS_on,
						const int version_major,
						const int version_minor,
						const int version_implement,
						const int version_build )
{
	writeBits ( e, profile,  4 );
	writeBits ( e, PNS_on,  1 );
	writeBits ( e, 0,  3 ); // unused
	writeBits ( e, version_major,  4 );
	writeBits ( e, version_minor,  8 );
	writeBits ( e, version_implement,  4 );
	writeBits ( e, version_build,  8 );
}

// formatting and writing SV7-bitstream for one frame
void
writeBitstream_SV8 ( mpc_encoder_t* e, int MaxBand, const SubbandQuantTyp*  Q )
{
	int n;
	const Huffman_t * Table, * Tables[2];

	for( n = MaxBand; n >= 0; n--)
		if (Res_L[n] != 0 || Res_R[n] != 0) break;

	n++;
	if (e->framesInBlock == 0) {
		encodeLog(e, n, MaxBand + 1);
		MaxBand = e->MaxBand = n;
	} else {
		n = n - e->MaxBand;
		MaxBand = e->MaxBand = n + e->MaxBand;
		if (n < 0) n += 33;
		writeBits(e, HuffBands[n].Code, HuffBands[n].Length);
	}

	/************************************ Resolution *********************************/

	if (MaxBand) {
		writeBits(e, HuffRes[0][Res_L[MaxBand - 1]].Code, HuffRes[0][Res_L[MaxBand - 1]].Length);
		writeBits(e, HuffRes[0][Res_R[MaxBand - 1]].Code, HuffRes[0][Res_R[MaxBand - 1]].Length);
		for ( n = MaxBand - 2; n >= 0; n--) {
			int tmp = Res_L[n] - Res_L[n + 1];
			if (tmp < 0) tmp += 17;
			writeBits(e, HuffRes[Res_L[n + 1] > 2][tmp].Code, HuffRes[Res_L[n + 1] > 2][tmp].Length);

			tmp = Res_R[n] - Res_R[n + 1];
			if (tmp < 0) tmp += 17;
			writeBits(e, HuffRes[Res_R[n + 1] > 2][tmp].Code, HuffRes[Res_R[n + 1] > 2][tmp].Length);
		}

		if (e->MS_Channelmode > 0) {
			mpc_uint32_t tmp = 0;
			int cnt = 0, tot = 0;
			for( n = 0; n < MaxBand; n++) {
				if ( Res_L[n] != 0 || Res_R[n] != 0 ) {
					tmp = (tmp << 1) | MS_Flag[n];
					cnt += MS_Flag[n];
					tot++;
				}
			}
			encodeLog(e, cnt, tot);
			if (cnt * 2 > tot) tmp = ~tmp;
			encodeEnum(e, tmp, tot);
		}
	}

	/************************************ SCF encoding type ***********************************/

	if (e->framesInBlock == 0){
		for( n = 0; n < 32; n++)
			DSCF_RLL_L[n] = DSCF_RLL_R[n] = 1; // new block -> force key frame
	}

	Tables[0] = HuffSCFI_1;
	Tables[1] = HuffSCFI_2;
	for ( n = 0; n < MaxBand; n++ ) {
		int tmp = 0, cnt = -1;
		if (Res_L[n]) {
			tmp = (SCF_Index_L[n][1] == SCF_Index_L[n][0]) * 2 + (SCF_Index_L[n][2] == SCF_Index_L[n][1]);
			cnt++;
		}
		if (Res_R[n]) {
			tmp = (tmp << 2) | ((SCF_Index_R[n][1] == SCF_Index_R[n][0]) * 2 + (SCF_Index_R[n][2] == SCF_Index_R[n][1]));
			cnt++;
		}
		if (cnt >= 0)
			writeBits(e, Tables[cnt][tmp].Code, Tables[cnt][tmp].Length);
	}

	/************************************* SCF **********************************/

	for ( n = 0; n < MaxBand; n++ ) {
		if ( Res_L[n] ) {
			int m;
			if (DSCF_RLL_L[n] == 1) {
				writeBits(e, SCF_Index_L[n][0] + 6, 7);
				DSCF_RLL_L[n] = 0;
			} else {
				unsigned int tmp = (SCF_Index_L[n][0] - SCF_Last_L[n] + 31) & 127;
				if (tmp < 64)
					writeBits(e, HuffDSCF_2[tmp].Code, HuffDSCF_2[tmp].Length);
				else {
					writeBits(e, HuffDSCF_2[64].Code, HuffDSCF_2[64].Length);
					writeBits(e, tmp - 64, 6);
				}
			}
			for( m = 0; m < 2; m++){
				if (SCF_Index_L[n][m+1] != SCF_Index_L[n][m]) {
					unsigned int tmp = (SCF_Index_L[n][m+1] - SCF_Index_L[n][m] + 31) & 127;
					if (tmp < 64)
						writeBits(e, HuffDSCF_1[tmp].Code, HuffDSCF_1[tmp].Length);
					else {
						writeBits(e, HuffDSCF_1[31].Code, HuffDSCF_1[31].Length);
						writeBits(e, tmp - 64, 6);
					}
				}
			}
			SCF_Last_L[n] = SCF_Index_L[n][2];
		}
		if ( Res_R[n] ) {
			int m;
			if (DSCF_RLL_R[n] == 1) {
				writeBits(e, SCF_Index_R[n][0] + 6, 7);
				DSCF_RLL_R[n] = 0;
			} else {
				unsigned int tmp = (SCF_Index_R[n][0] - SCF_Last_R[n] + 31) & 127;
				if (tmp < 64)
					writeBits(e, HuffDSCF_2[tmp].Code, HuffDSCF_2[tmp].Length);
				else {
					writeBits(e, HuffDSCF_2[64].Code, HuffDSCF_2[64].Length);
					writeBits(e, tmp - 64, 6);
				}
			}
			for( m = 0; m < 2; m++){
				if (SCF_Index_R[n][m+1] != SCF_Index_R[n][m]) {
					unsigned int tmp = (SCF_Index_R[n][m+1] - SCF_Index_R[n][m] + 31) & 127;
					if (tmp < 64)
						writeBits(e, HuffDSCF_1[tmp].Code, HuffDSCF_1[tmp].Length);
					else {
						writeBits(e, HuffDSCF_1[31].Code, HuffDSCF_1[31].Length);
						writeBits(e, tmp - 64, 6);
					}
				}
			}
			SCF_Last_R[n] = SCF_Index_R[n][2];
		}
	}

	/*********************************** Samples *********************************/
	for ( n = 0; n < MaxBand; n++ ) {
		int Res = Res_L[n];
		const unsigned int * q = Q[n].L;
		static const unsigned int thres[] = {0, 0, 3, 7, 9, 1, 3, 4, 8};
		static const int HuffQ2_var[5*5*5] =
		{6, 5, 4, 5, 6, 5, 4, 3, 4, 5, 4, 3, 2, 3, 4, 5, 4, 3, 4, 5, 6, 5, 4, 5, 6, 5, 4, 3, 4, 5, 4, 3, 2, 3, 4, 3, 2, 1, 2, 3, 4, 3, 2, 3, 4, 5, 4, 3, 4, 5, 4, 3, 2, 3, 4, 3, 2, 1, 2, 3, 2, 1, 0, 1, 2, 3, 2, 1, 2, 3, 4, 3, 2, 3, 4, 5, 4, 3, 4, 5, 4, 3, 2, 3, 4, 3, 2, 1, 2, 3, 4, 3, 2, 3, 4, 5, 4, 3, 4, 5, 6, 5, 4, 5, 6, 5, 4, 3, 4, 5, 4, 3, 2, 3, 4, 5, 4, 3, 4, 5, 6, 5, 4, 5, 6};

		do {
			int k = 0, idx = 1, cnt = 0, sng;
			switch ( Res ) {
				case -1:
				case  0:
					break;
				case  1:
					Table = HuffQ [0][0];
					for( ; k < 36; ){
						int kmax = k + 18;
						cnt = 0, sng = 0;
						for ( ; k < kmax; k++) {
							idx <<= 1;
							if (q[k] != 1) {
								cnt++;
								idx |= 1;
								sng = (sng << 1) | (q[k] >> 1);
							}
						}
						writeBits(e, Table[cnt].Code, Table[cnt].Length);
						if (cnt > 0) {
							if (cnt > 9) idx = ~idx;
							encodeEnum(e, idx, 18);
							writeBits(e, sng, cnt);
						}
					}
					break;
				case  2:
					Tables[0] = HuffQ [0][1];
					Tables[1] = HuffQ [1][1];
					idx = 2 * thres[Res];
					for ( ; k < 36; k += 3) {
						int tmp = q[k] + 5*q[k+1] + 25*q[k+2];
						writeBits ( e, Tables[idx > thres[Res]][tmp].Code,
									Tables[idx > thres[Res]][tmp].Length );
						idx = (idx >> 1) + HuffQ2_var[tmp];
					}
					break;
				case  3:
				case  4:
					Table = HuffQ [0][Res - 1];
					for ( ; k < 36; k += 2 ) {
						int tmp = q[k] + thres[Res]*q[k+1];
						writeBits ( e, Table[tmp].Code, Table[tmp].Length );
					}
					break;
				case  5:
				case  6:
				case  7:
				case  8:
					Tables[0] = HuffQ [0][Res - 1];
					Tables[1] = HuffQ [1][Res - 1];
					idx = 2 * thres[Res];
					for ( ; k < 36; k++ ) {
						int tmp = q[k] - (1 << (Res - 2)) + 1;
						writeBits ( e, Tables[idx > thres[Res]][q[k]].Code,
									Tables[idx > thres[Res]][q[k]].Length );
						if (tmp < 0) tmp = -tmp;
						idx = (idx >> 1) + tmp;
					}
					break;
				default:
					for ( ; k < 36; k++ ) {
						writeBits ( e, HuffQ9up[q[k] >> (Res - 9)].Code,
									HuffQ9up[q[k] >> (Res - 9)].Length );
						if (Res != 9)
							writeBits ( e, q[k] & ((1 << (Res - 9)) - 1), Res - 9);
					}
					break;
			}

			Res = Res_R[n];
		} while (q == Q[n].L && (q = Q[n].R));
	}

	e->framesInBlock++;
	if (e->framesInBlock == FRAMES_PER_BLOCK) {
		if ((e->block_cnt & ((1 << e->seek_pwr) - 1)) == 0) {
			e->seek_table[e->seek_pos] = ftell(e->outputFile);
			e->seek_pos++;
		}
		e->block_cnt++;
		writeBlock(e, "AD", MPC_FALSE);
	}
}

#if 0

typedef struct {
	int Symbol;
	unsigned int Count;
	unsigned int Code;
	unsigned int Bits;
} huff_sym_t;

void _Huffman_MakeTree( huff_sym_t *sym, unsigned int num_symbols);
void _Huffman_PrintCodes(huff_sym_t * sym, unsigned int num_symbols, int print_type, int offset);

void print_histo(void)
{
	int i, j;
	huff_sym_t sym[HISTO_NB][HISTO_LEN];
	unsigned int dist[HISTO_NB];
	unsigned int size[HISTO_NB];
	unsigned int cnt[HISTO_NB];
	unsigned int total_cnt, total_size, full_count = 0, full_size = 0;
	double optim_size, full_optim = 0;

	return;

	memset(dist, 1, sizeof dist);
	memset(sym, 0, sizeof(huff_sym_t) * HISTO_LEN * HISTO_NB);

	for(j = 0 ; j < HISTO_NB ; j++) {
		for(i = 0 ; i < HISTO_LEN; i++) {
			sym[j][i].Symbol = i;
			sym[j][i].Count = histo[j][i];
			if (sym[j][i].Count == 0)
				sym[j][i].Count = 1;
		}
		_Huffman_MakeTree(sym[j], HISTO_LEN);
		_Huffman_PrintCodes(sym[j], HISTO_LEN, 3, 0);
		_Huffman_PrintCodes(sym[j], HISTO_LEN, 0, 0);
		_Huffman_PrintCodes(sym[j], HISTO_LEN, 1, 0);
		total_cnt = 0;
		total_size = 0;
		optim_size = 0;
		for( i = 0; i < HISTO_LEN; i++) {
			total_cnt += sym[j][i].Count;
			total_size += sym[j][i].Count * sym[j][i].Bits;
			if (sym[j][i].Count != 0)
				optim_size += sym[j][i].Count * __builtin_log2(sym[j][i].Count);
		}
		full_count += total_cnt;
		full_size += total_size;
		optim_size = total_cnt * __builtin_log2(total_cnt) - optim_size;
		full_optim += optim_size;
		size[j] = total_size;
		cnt[j] = total_cnt;
		printf("%u count : %u huff : %f bps ", j, total_cnt, (float)total_size / total_cnt);
		printf("opt : %f bps ", (float)optim_size / total_cnt);
		printf("loss : %f bps (%f %%)\n", (float)(total_size - optim_size) / total_cnt, (float)(total_size - optim_size) * 100 / optim_size);
		for( i = 0; i < HISTO_LEN; i++){
			printf("%u ", sym[j][i].Bits);
		}

		printf("\n\n");
	}
	printf("cnt : %u size %f optim %f\n", full_count, (float)full_size / full_count, (float)full_optim / full_count);
	printf("loss : %f bps (%f %%)\n", (float)(full_size - full_optim) / full_count, (float)(full_size - full_optim) * 100 / full_optim);


	printf("\n");
}

void
Dump ( const unsigned int* q, const int Res )
{
    switch ( Res ) {
    case  1:
        for ( k = 0; k < 36; k++, q++ )
            printf ("%2d%c", *q-1, k==35?'\n':' ');
        break;
    case  2:
        for ( k = 0; k < 36; k++, q++ )
            printf ("%2d%c", *q-2, k==35?'\n':' ');
        break;
    case  3: case  4: case  5: case  6: case  7:
        if ( Res == 5 )
            for ( k = 0; k < 36; k++, q++ )
                printf ("%2d%c", *q-7, k==35?'\n':' ');
        break;
    case  8: case  9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
        printf ("%2u: ", Res-1 );
        for ( k = 0; k < 36; k++, q++ ) {
            printf ("%6d", *q - (1 << (Res-2)) );
        }
        printf ("\n");
        break;
    }
}
#endif

/* end of encode_sv7.c */
