/*
 * Copyright (c) 2005, The Musepack Development Team
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *
 *     * Redistributions in binary form must reproduce the above
 *       copyright notice, this list of conditions and the following
 *       disclaimer in the documentation and/or other materials provided
 *       with the distribution.
 *
 *     * Neither the name of the The Musepack Development Team nor the
 *       names of its contributors may be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "libmpc.h"

#define FORCED_THREAD_STACKSIZE 1024 * 1000


using TagLib::MPC::File;
using TagLib::Tag;
using TagLib::String;
using TagLib::APE::ItemListMap;

InputPlugin MpcPlugin = {
    NULL,           //File Handle               FILE* handle
    NULL,           //Filename                  char* filename
    NULL,           //Name of Plugin            char* filename
    mpcOpenPlugin,  //Open Plugin               [CALLBACK]
    mpcAboutBox,    //Show About box            [CALLBACK]
    mpcConfigBox,   //Show Configure box        [CALLBACK]
    mpcIsOurFile,   //Check if it's our file    [CALLBACK]
    NULL,           //Scan the directory        [UNUSED]
    mpcPlay,        //Play                      [CALLBACK]
    mpcStop,        //Stop                      [CALLBACK]
    mpcPause,       //Pause                     [CALLBACK]
    mpcSeek,        //Seek                      [CALLBACK]
    mpcSetEq,       //Set EQ                    [CALLBACK]
    mpcGetTime,     //Get Time                  [CALLBACK]
    NULL,           //Get Volume                [UNUSED]
    NULL,           //Set Volume                [UNUSED]
    mpcClosePlugin, //Close Plugin              [CALLBACK]
    NULL,           //Obsolete                  [UNUSED]
    NULL,           //Visual plugins            add_vis_pcm(int time, AFormat fmt, int nch, int length, void *ptr)
    NULL,           //Set Info Settings         set_info(char *title, int length, int rate, int freq, int nch)
    NULL,           //set Info Text             set_info_text(char* text)
    mpcGetSongInfo, //Get Title String callback [CALLBACK]
    mpcFileInfoBox, //Show File Info Box        [CALLBACK]
    NULL,           //Output Plugin Handle      OutputPlugin output
};

extern "C"
InputPlugin* get_iplugin_info()
{
    MpcPlugin.description = g_strdup_printf("Musepack Audio Plugin %s", VERSION);
    return &MpcPlugin;
}

static PluginConfig pluginConfig = {0};
static Widgets      widgets      = {0};
static MpcDecoder   mpcDecoder   = {0};
static TrackInfo    track        = {0};

static pthread_t           threadHandle;
static pthread_attr_t      threadAttr;
static pthread_mutex_t     threadMutex;
static pthread_mutexattr_t threadMutexAttr;

static void mpcOpenPlugin()
{
    pthread_mutexattr_init(&threadMutexAttr);
    pthread_mutex_init(&threadMutex, &threadMutexAttr);
    pthread_attr_init(&threadAttr);
    pthread_attr_setstacksize(&threadAttr, FORCED_THREAD_STACKSIZE);
    pluginConfig.iFromUTF8 = iconv_open("//IGNORE", "UTF8");
    pluginConfig.iToUTF8 = iconv_open("UTF8//IGNORE", "");

    ConfigFile *cfg;
    cfg = xmms_cfg_open_default_file();
    xmms_cfg_read_boolean(cfg, "musepack", "clipPrevention", &pluginConfig.clipPrevention);
    xmms_cfg_read_boolean(cfg, "musepack", "albumGain",      &pluginConfig.albumGain);
    xmms_cfg_read_boolean(cfg, "musepack", "dynamicBitrate", &pluginConfig.dynamicBitrate);
    xmms_cfg_read_boolean(cfg, "musepack", "replaygain",     &pluginConfig.replaygain);
    xmms_cfg_read_boolean(cfg, "musepack", "fastSeek",       &pluginConfig.fastSeek);
    xmms_cfg_free(cfg);
}

static void mpcAboutBox()
{
    GtkWidget* aboutBox = widgets.aboutBox;
    if (aboutBox)
        gdk_window_raise(aboutBox->window);
    else
    {
        char* titleText      = g_strdup_printf("Musepack Decoder Plugin %s", VERSION);
        char* contentText = "Plugin code by\nBenoit Amiaux\nMartin Spuler\nKuniklo\n\nGet latest version at http://musepack.net\n";
        char* buttonText  = "Nevermind";
        aboutBox = xmms_show_message(titleText, contentText, buttonText, FALSE, NULL, NULL);
        widgets.aboutBox = aboutBox;
        gtk_signal_connect(GTK_OBJECT(aboutBox), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &widgets.aboutBox);
    }
}

static void mpcConfigBox()
{
    GtkWidget* configBox = widgets.configBox;
    if(configBox)
        gdk_window_raise(configBox->window);
    else
    {
        configBox = gtk_window_new(GTK_WINDOW_DIALOG);
        widgets.configBox = configBox;
        gtk_signal_connect(GTK_OBJECT(configBox), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &widgets.configBox);
        gtk_window_set_title(GTK_WINDOW(configBox), "Musepack Decoder Configuration");
        gtk_window_set_policy(GTK_WINDOW(configBox), FALSE, FALSE, FALSE);
        gtk_container_border_width(GTK_CONTAINER(configBox), 10);

        GtkWidget* notebook = gtk_notebook_new();
        GtkWidget* vbox = gtk_vbox_new(FALSE, 10);
        gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
        gtk_container_add(GTK_CONTAINER(configBox), vbox);

        //General Settings Tab
        GtkWidget* generalSet = gtk_frame_new("General Settings");
        gtk_container_border_width(GTK_CONTAINER(generalSet), 5);

        GtkWidget* gSvbox = gtk_vbox_new(FALSE, 10);
        gtk_container_border_width(GTK_CONTAINER(gSvbox), 5);
        gtk_container_add(GTK_CONTAINER(generalSet), gSvbox);

        GtkWidget* bitrateCheck = gtk_check_button_new_with_label("Enable Dynamic Bitrate Display");
        widgets.bitrateCheck = bitrateCheck;
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bitrateCheck), pluginConfig.dynamicBitrate);
        gtk_box_pack_start(GTK_BOX(gSvbox), bitrateCheck, FALSE, FALSE, 0);

        GtkWidget* seekCheck = gtk_check_button_new_with_label("Enable Fast Seeking");
        widgets.seekCheck = seekCheck;
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(seekCheck), pluginConfig.fastSeek);
        gtk_box_pack_start(GTK_BOX(gSvbox), seekCheck, FALSE, FALSE, 0);
        gtk_notebook_append_page(GTK_NOTEBOOK(notebook), generalSet, gtk_label_new("Plugin"));

        //ReplayGain Settings Tab
        GtkWidget* replaygainSet = gtk_frame_new("ReplayGain Settings");
        gtk_container_border_width(GTK_CONTAINER(replaygainSet), 5);

        GtkWidget* rSVbox = gtk_vbox_new(FALSE, 10);
        gtk_container_border_width(GTK_CONTAINER(rSVbox), 5);
        gtk_container_add(GTK_CONTAINER(replaygainSet), rSVbox);

        GtkWidget* clippingCheck = gtk_check_button_new_with_label("Enable Clipping Prevention");
        widgets.clippingCheck = clippingCheck;
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(clippingCheck), pluginConfig.clipPrevention);
        gtk_box_pack_start(GTK_BOX(rSVbox), clippingCheck, FALSE, FALSE, 0);

        GtkWidget* replaygainCheck = gtk_check_button_new_with_label("Enable ReplayGain");
        widgets.replaygainCheck = replaygainCheck;
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(replaygainCheck), pluginConfig.replaygain);
        gtk_box_pack_start(GTK_BOX(rSVbox), replaygainCheck, FALSE, FALSE, 0);

        GtkWidget* replaygainType = gtk_frame_new("ReplayGain Type");
        gtk_box_pack_start(GTK_BOX(rSVbox), replaygainType, FALSE, FALSE, 0);
        gtk_signal_connect(GTK_OBJECT(replaygainCheck), "toggled", GTK_SIGNAL_FUNC(toggleSwitch), replaygainType);

        GtkWidget* rgVbox = gtk_vbox_new(FALSE, 5);
        gtk_container_set_border_width(GTK_CONTAINER(rgVbox), 5);
        gtk_container_add(GTK_CONTAINER(replaygainType), rgVbox);

        GtkWidget* trackCheck = gtk_radio_button_new_with_label(NULL, "Use Track Gain");
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(trackCheck), !pluginConfig.albumGain);
        gtk_box_pack_start(GTK_BOX(rgVbox), trackCheck, FALSE, FALSE, 0);

        GtkWidget* albumCheck = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(trackCheck)), "Use Album Gain");
        widgets.albumCheck = albumCheck;
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(albumCheck), pluginConfig.albumGain);
        gtk_box_pack_start(GTK_BOX(rgVbox), albumCheck, FALSE, FALSE, 0);
        gtk_widget_set_sensitive(replaygainType, pluginConfig.replaygain);
        gtk_notebook_append_page(GTK_NOTEBOOK(notebook), replaygainSet, gtk_label_new("ReplayGain"));

        //Buttons
        GtkWidget* buttonBox = gtk_hbutton_box_new();
        gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonBox), GTK_BUTTONBOX_END);
        gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonBox), 5);
        gtk_box_pack_start(GTK_BOX(vbox), buttonBox, FALSE, FALSE, 0);

        GtkWidget* okButton = gtk_button_new_with_label("Ok");
        gtk_signal_connect(GTK_OBJECT(okButton), "clicked", GTK_SIGNAL_FUNC(saveConfigBox), NULL);
        GTK_WIDGET_SET_FLAGS(okButton, GTK_CAN_DEFAULT);
        gtk_box_pack_start(GTK_BOX(buttonBox), okButton, TRUE, TRUE, 0);

        GtkWidget* cancelButton = gtk_button_new_with_label("Cancel");
        gtk_signal_connect_object(GTK_OBJECT(cancelButton), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(widgets.configBox));
        GTK_WIDGET_SET_FLAGS(cancelButton, GTK_CAN_DEFAULT);
        gtk_widget_grab_default(cancelButton);
        gtk_box_pack_start(GTK_BOX(buttonBox), cancelButton, TRUE, TRUE, 0);

        gtk_widget_show_all(configBox);
    }
}

static void toggleSwitch(GtkWidget* p_Widget, gpointer p_Data)
{
    gtk_widget_set_sensitive(GTK_WIDGET(p_Data), gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(p_Widget)));
}

static void saveConfigBox(GtkWidget* p_Widget, gpointer p_Data)
{
    ConfigFile* cfg;
    GtkToggleButton* tb;

    tb = GTK_TOGGLE_BUTTON(widgets.replaygainCheck);
    pluginConfig.replaygain = gtk_toggle_button_get_active(tb);
    tb = GTK_TOGGLE_BUTTON(widgets.clippingCheck);
    pluginConfig.clipPrevention = gtk_toggle_button_get_active(tb);
    tb = GTK_TOGGLE_BUTTON(widgets.bitrateCheck);
    pluginConfig.dynamicBitrate = gtk_toggle_button_get_active(tb);
    tb = GTK_TOGGLE_BUTTON(widgets.seekCheck);
    pluginConfig.fastSeek = gtk_toggle_button_get_active(tb);
    tb = GTK_TOGGLE_BUTTON(widgets.albumCheck);
    pluginConfig.albumGain = gtk_toggle_button_get_active(tb);

    cfg = xmms_cfg_open_default_file();

    xmms_cfg_write_boolean (cfg, "musepack", "clipPrevention", pluginConfig.clipPrevention);
    xmms_cfg_write_boolean (cfg, "musepack", "albumGain",      pluginConfig.albumGain);
    xmms_cfg_write_boolean (cfg, "musepack", "dynamicBitrate", pluginConfig.dynamicBitrate);
    xmms_cfg_write_boolean (cfg, "musepack", "replaygain",     pluginConfig.replaygain);
    xmms_cfg_write_boolean (cfg, "musepack", "fastSeek",       pluginConfig.fastSeek);
    xmms_cfg_write_default_file (cfg);

    xmms_cfg_free (cfg);

    gtk_widget_destroy (widgets.configBox);
}

static int mpcIsOurFile(char* p_Filename)
{
    char* ext;
    ext = strrchr(p_Filename, '.');
    if(ext)
        return !(strcasecmp(ext, ".mpc") && strcasecmp(ext, ".mpp") && strcasecmp(ext, ".mp+"));
    return FALSE;
}

static void mpcPlay(char* p_Filename)
{
    mpcDecoder.offset   = -1;
    mpcDecoder.isAlive  = true;
    mpcDecoder.isOutput = false;
    mpcDecoder.isPause  = false;
    pthread_create(&threadHandle, &threadAttr, decodeStream, (void *) p_Filename);
}

static void mpcStop()
{
    setAlive(false);
    if (threadHandle)
    {
        pthread_join(threadHandle, NULL);
        if (mpcDecoder.isOutput)
        {
            MpcPlugin.output->buffer_free();
            MpcPlugin.output->close_audio();
            mpcDecoder.isOutput = false;
        }
    }
}

static void mpcPause(short p_Pause)
{
    lockAcquire();
    mpcDecoder.isPause = p_Pause;
    MpcPlugin.output->pause(p_Pause);
    lockRelease();
}

static void mpcSeek(int p_Offset)
{
    lockAcquire();
    mpcDecoder.offset = static_cast<double> (p_Offset);
    MpcPlugin.output->flush(1000 * p_Offset);
    lockRelease();
}

static void mpcSetEq(int on, float preamp, float* eq)
{
    pluginConfig.isEq = static_cast<bool> (on);
    init_iir(on, preamp, eq);
}

static int mpcGetTime()
{
    if(!isAlive())
        return -1;
    return MpcPlugin.output->output_time();
}

static void mpcClosePlugin()
{
    pthread_mutex_destroy(&threadMutex);
    pthread_mutexattr_destroy(&threadMutexAttr);
    pthread_attr_destroy(&threadAttr);
    iconv_close(pluginConfig.iFromUTF8);
    iconv_close(pluginConfig.iToUTF8);
}

static void mpcGetSongInfo(char* p_Filename, char** p_Title, int* p_Length)
{
	mpc_reader reader;
	mpc_demux * demux;
	if (mpc_reader_init_stdio(&reader, p_Filename) == MPC_STATUS_OK && (demux = mpc_demux_init(&reader))) {
        MpcInfo tags = getTags(p_Filename);
        *p_Title = mpcGenerateTitle(tags, p_Filename);
        freeTags(tags);
		mpc_streaminfo info;
		mpc_demux_get_info(demux, &info);
		*p_Length = static_cast<int> (1000 * mpc_streaminfo_get_length(&info));
		mpc_demux_exit(demux);
		mpc_reader_exit_stdio(&reader);
    } else {
        char* temp = g_strdup_printf("[xmms-musepack] mpcGetSongInfo is unable to open %s\n", p_Filename);
        perror(temp);
        free(temp);
    }
}

static void freeTags(MpcInfo& tags)
{
    free(tags.title);
    free(tags.artist);
    free(tags.album);
    free(tags.comment);
    free(tags.genre);
    free(tags.date);
}

static char* convertCode(const char* text, iconv_t& code)
{
    if (code == (iconv_t) -1) {
        perror("[xmms-musepack] convertFromUTF8 is unable to open Iconv descriptor");
        return g_strdup(text);
    }

    size_t inSize  = strlen(text);
    size_t outSize = 2 * inSize + 1;
    char* outBuf = (char*) malloc(outSize);
#ifdef __FreeBSD__
    const char* in = text;
#else
    char* in = (char*) text;
#endif
    char* out = outBuf;
    memset(outBuf, 0, outSize);
    iconv(code, &in, &inSize, &out, &outSize);

    if(inSize)
    {
        char* temp = g_strdup_printf("[xmms-musepack] invalid unicode sequence detected in %s", text);
        perror(temp);
        free(temp);
    }

    return outBuf;
}

static MpcInfo getTags(const char* p_Filename)
{
    File oFile(p_Filename, false);
    Tag* poTag = oFile.tag();
    MpcInfo tags = {0};
    tags.title   = convertCode(poTag->title().toCString(true),   pluginConfig.iFromUTF8);
    tags.artist  = convertCode(poTag->artist().toCString(true),  pluginConfig.iFromUTF8);
    tags.album   = convertCode(poTag->album().toCString(true),   pluginConfig.iFromUTF8);
    tags.genre   = convertCode(poTag->genre().toCString(true),   pluginConfig.iFromUTF8);
    tags.comment = convertCode(poTag->comment().toCString(true), pluginConfig.iFromUTF8);
    tags.year    = poTag->year();
    tags.track   = poTag->track();
    TagLib::APE::Tag* ape = oFile.APETag(false);
    if(ape)
    {
        ItemListMap map = ape->itemListMap();
        if(map.contains("YEAR"))
        {
            tags.date = convertCode(map["YEAR"].toString().toCString(true), pluginConfig.iFromUTF8);
        }
        else
        {
            tags.date = g_strdup_printf("%d", tags.year);
        }
    }
    return tags;
}

static void mpcFileInfoBox(char* p_Filename)
{
    GtkWidget* infoBox = widgets.infoBox;

    if(infoBox)
        gdk_window_raise(infoBox->window);
    else
    {
        infoBox = gtk_window_new(GTK_WINDOW_DIALOG);
        widgets.infoBox = infoBox;
        gtk_window_set_policy(GTK_WINDOW(infoBox), FALSE, FALSE, FALSE);
        gtk_signal_connect(GTK_OBJECT(infoBox), "destroy", GTK_SIGNAL_FUNC(closeInfoBox), NULL);
        gtk_container_set_border_width(GTK_CONTAINER(infoBox), 10);

        GtkWidget* iVbox = gtk_vbox_new(FALSE, 10);
        gtk_container_add(GTK_CONTAINER(infoBox), iVbox);

        GtkWidget* filenameHbox = gtk_hbox_new(FALSE, 5);
        gtk_box_pack_start(GTK_BOX(iVbox), filenameHbox, FALSE, TRUE, 0);

        GtkWidget* fileLabel = gtk_label_new("Filename:");
        gtk_box_pack_start(GTK_BOX(filenameHbox), fileLabel, FALSE, TRUE, 0);

        GtkWidget* fileEntry = gtk_entry_new();
        widgets.fileEntry = fileEntry;
        gtk_editable_set_editable(GTK_EDITABLE(fileEntry), FALSE);
        gtk_box_pack_start(GTK_BOX(filenameHbox), fileEntry, TRUE, TRUE, 0);

        GtkWidget* iHbox = gtk_hbox_new(FALSE, 10);
        gtk_box_pack_start(GTK_BOX(iVbox), iHbox, FALSE, TRUE, 0);

        GtkWidget* leftBox = gtk_vbox_new(FALSE, 10);
        gtk_box_pack_start(GTK_BOX(iHbox), leftBox, FALSE, FALSE, 0);

        //Tag labels
        GtkWidget* tagFrame = gtk_frame_new("Musepack Tag");
        gtk_box_pack_start(GTK_BOX(leftBox), tagFrame, FALSE, FALSE, 0);
        gtk_widget_set_sensitive(tagFrame, TRUE);

        GtkWidget* iTable = gtk_table_new(5, 5, FALSE);
        gtk_container_set_border_width(GTK_CONTAINER(iTable), 5);
        gtk_container_add(GTK_CONTAINER(tagFrame), iTable);

        mpcGtkTagLabel("Title:", 0, 1, 0, 1, iTable);
        GtkWidget* titleEntry = mpcGtkTagEntry(1, 4, 0, 1, 0, iTable);
        widgets.titleEntry = titleEntry;

        mpcGtkTagLabel("Artist:", 0, 1, 1, 2, iTable);
        GtkWidget* artistEntry = mpcGtkTagEntry(1, 4, 1, 2, 0, iTable);
        widgets.artistEntry = artistEntry;

        mpcGtkTagLabel("Album:", 0, 1, 2, 3, iTable);
        GtkWidget* albumEntry = mpcGtkTagEntry(1, 4, 2, 3, 0, iTable);
        widgets.albumEntry = albumEntry;

        mpcGtkTagLabel("Comment:", 0, 1, 3, 4, iTable);
        GtkWidget* commentEntry = mpcGtkTagEntry(1, 4, 3, 4, 0, iTable);
        widgets.commentEntry = commentEntry;

        mpcGtkTagLabel("Year:", 0, 1, 4, 5, iTable);
        GtkWidget* yearEntry = mpcGtkTagEntry(1, 2, 4, 5, 4, iTable);
        widgets.yearEntry = yearEntry;
        gtk_widget_set_usize(yearEntry, 4, -1);

        mpcGtkTagLabel("Track:", 2, 3, 4, 5, iTable);
        GtkWidget* trackEntry = mpcGtkTagEntry(3, 4, 4, 5, 4, iTable);
        widgets.trackEntry = trackEntry;
        gtk_widget_set_usize(trackEntry, 3, -1);

        mpcGtkTagLabel("Genre:", 0, 1, 5, 6, iTable);
        GtkWidget* genreEntry = mpcGtkTagEntry(1, 4, 5, 6, 0, iTable);
        widgets.genreEntry = genreEntry;
        gtk_widget_set_usize(genreEntry, 20, -1);

        //Buttons
        GtkWidget* buttonBox = gtk_hbutton_box_new();
        gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonBox), GTK_BUTTONBOX_END);
        gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttonBox), 5);
        gtk_box_pack_start(GTK_BOX(leftBox), buttonBox, FALSE, FALSE, 0);

        GtkWidget* saveButton = mpcGtkButton("Save", buttonBox);
        gtk_signal_connect(GTK_OBJECT(saveButton), "clicked", GTK_SIGNAL_FUNC(saveTags), NULL);

        GtkWidget* removeButton = mpcGtkButton("Remove Tag", buttonBox);
        gtk_signal_connect_object(GTK_OBJECT(removeButton), "clicked", GTK_SIGNAL_FUNC(removeTags), NULL);

        GtkWidget* cancelButton = mpcGtkButton("Cancel", buttonBox);
        gtk_signal_connect_object(GTK_OBJECT(cancelButton), "clicked", GTK_SIGNAL_FUNC(closeInfoBox), NULL);
        gtk_widget_grab_default(cancelButton);

        //File information
        GtkWidget* infoFrame = gtk_frame_new("Musepack Info");
        gtk_box_pack_start(GTK_BOX(iHbox), infoFrame, FALSE, FALSE, 0);

        GtkWidget* infoVbox = gtk_vbox_new(FALSE, 5);
        gtk_container_add(GTK_CONTAINER(infoFrame), infoVbox);
        gtk_container_set_border_width(GTK_CONTAINER(infoVbox), 10);
        gtk_box_set_spacing(GTK_BOX(infoVbox), 0);

        GtkWidget* streamLabel    = mpcGtkLabel(infoVbox);
        GtkWidget* encoderLabel   = mpcGtkLabel(infoVbox);
		GtkWidget* profileLabel   = mpcGtkLabel(infoVbox);
		GtkWidget* pnsLabel       = mpcGtkLabel(infoVbox);
		GtkWidget* gaplessLabel   = mpcGtkLabel(infoVbox);
        GtkWidget* bitrateLabel   = mpcGtkLabel(infoVbox);
        GtkWidget* rateLabel      = mpcGtkLabel(infoVbox);
        GtkWidget* channelsLabel  = mpcGtkLabel(infoVbox);
        GtkWidget* lengthLabel    = mpcGtkLabel(infoVbox);
        GtkWidget* fileSizeLabel  = mpcGtkLabel(infoVbox);
        GtkWidget* trackPeakLabel = mpcGtkLabel(infoVbox);
        GtkWidget* trackGainLabel = mpcGtkLabel(infoVbox);
        GtkWidget* albumPeakLabel = mpcGtkLabel(infoVbox);
        GtkWidget* albumGainLabel = mpcGtkLabel(infoVbox);

		mpc_reader reader;
		mpc_demux * demux;
		if (mpc_reader_init_stdio(&reader, p_Filename) == MPC_STATUS_OK && (demux = mpc_demux_init(&reader))) {
            mpc_streaminfo info;
			mpc_demux_get_info(demux, &info);

            int time = static_cast<int> (mpc_streaminfo_get_length(&info));
            int minutes = time / 60;
            int seconds = time % 60;

            mpcGtkPrintLabel(streamLabel,    "Streamversion %d", info.stream_version);
            mpcGtkPrintLabel(encoderLabel,   "Encoder: \%s", info.encoder);
			mpcGtkPrintLabel(profileLabel,   "Profile: \%s (q=%0.2f)", info.profile_name, info.profile - 5);
			mpcGtkPrintLabel(pnsLabel,       "PNS: \%s", info.pns ? "on" : "off");
			mpcGtkPrintLabel(gaplessLabel,   "Gapless: \%s", info.is_true_gapless ? "on" : "off");
            mpcGtkPrintLabel(bitrateLabel,   "Average bitrate: \%6.1f kbps", info.average_bitrate * 1.e-3);
            mpcGtkPrintLabel(rateLabel,      "Samplerate: \%d Hz", info.sample_freq);
            mpcGtkPrintLabel(channelsLabel,  "Channels: \%d", info.channels);
            mpcGtkPrintLabel(lengthLabel,    "Length: \%d:\%.2d", minutes, seconds);
            mpcGtkPrintLabel(fileSizeLabel,  "File size: \%d Bytes", info.total_file_length);
			mpcGtkPrintLabel(trackPeakLabel, "Track Peak: \%2.2f dB", info.peak_title / 256.);
			mpcGtkPrintLabel(trackGainLabel, "Track Gain: \%2.2f dB", info.gain_title / 256.);
			mpcGtkPrintLabel(albumPeakLabel, "Album Peak: \%2.2f dB", info.peak_album / 256.);
			mpcGtkPrintLabel(albumGainLabel, "Album Gain: \%2.2f dB", info.gain_album / 256.);

            MpcInfo tags = getTags(p_Filename);
            gtk_entry_set_text(GTK_ENTRY(titleEntry),   tags.title);
            gtk_entry_set_text(GTK_ENTRY(artistEntry),  tags.artist);
            gtk_entry_set_text(GTK_ENTRY(albumEntry),   tags.album);
            gtk_entry_set_text(GTK_ENTRY(commentEntry), tags.comment);
            gtk_entry_set_text(GTK_ENTRY(genreEntry),   tags.genre);
            char* entry = g_strdup_printf ("%d", tags.track);
            gtk_entry_set_text(GTK_ENTRY(trackEntry), entry);
            free(entry);
            entry = g_strdup_printf ("%d", tags.year);
            gtk_entry_set_text(GTK_ENTRY(yearEntry), entry);
            free(entry);
            entry = convertCode(p_Filename, pluginConfig.iFromUTF8);
            gtk_entry_set_text(GTK_ENTRY(fileEntry), entry);
            free(entry);
            freeTags(tags);
			mpc_demux_exit(demux);
			mpc_reader_exit_stdio(&reader);
        }
        else
        {
            char* temp = g_strdup_printf("[xmms-musepack] mpcFileInfoBox is unable to read tags from %s", p_Filename);
            perror(temp);
            free(temp);
        }

        char* name = convertCode(p_Filename, pluginConfig.iFromUTF8);
        char* text = g_strdup_printf("File Info - %s", g_basename(name));
        gtk_window_set_title(GTK_WINDOW(infoBox), text);
        free(name);
        free(text);

        gtk_widget_show_all(infoBox);
    }
}

static void mpcGtkPrintLabel(GtkWidget* widget, char* format,...)
{
    va_list args;

    va_start(args, format);
    char* temp = g_strdup_vprintf(format, args);
    va_end(args);

    gtk_label_set_text(GTK_LABEL(widget), temp);
    free(temp);
}

static GtkWidget* mpcGtkTagLabel(char* p_Text, int a, int b, int c, int d, GtkWidget* p_Box)
{
    GtkWidget* label = gtk_label_new(p_Text);
    gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);
    gtk_table_attach(GTK_TABLE(p_Box), label, a, b, c, d, GTK_FILL, GTK_FILL, 5, 5);
    return label;
}

static GtkWidget* mpcGtkTagEntry(int a, int b, int c, int d, int p_Size, GtkWidget* p_Box)
{
    GtkWidget* entry;
    if(p_Size == 0)
        entry = gtk_entry_new();
    else
        entry = gtk_entry_new_with_max_length(p_Size);
    gtk_table_attach(GTK_TABLE(p_Box), entry, a, b, c, d,
                    (GtkAttachOptions) (GTK_FILL | GTK_EXPAND | GTK_SHRINK),
                    (GtkAttachOptions) (GTK_FILL | GTK_EXPAND | GTK_SHRINK), 0, 5);
    return entry;
}

static GtkWidget* mpcGtkLabel(GtkWidget* p_Box)
{
    GtkWidget* label = gtk_label_new("");
    gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
    gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
    gtk_box_pack_start(GTK_BOX(p_Box), label, FALSE, FALSE, 0);
    return label;
}

static GtkWidget* mpcGtkButton(char* p_Text, GtkWidget* p_Box)
{
    GtkWidget* button = gtk_button_new_with_label(p_Text);
    GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(p_Box), button, TRUE, TRUE, 0);
    return button;
}

static void removeTags(GtkWidget * w, gpointer data)
{
    File oFile(gtk_entry_get_text(GTK_ENTRY(widgets.fileEntry)));
    oFile.remove();
    oFile.save();
    closeInfoBox(NULL, NULL);
}

static void saveTags(GtkWidget* w, gpointer data)
{
    File oFile(gtk_entry_get_text(GTK_ENTRY(widgets.fileEntry)));
    Tag* poTag = oFile.tag();

    char* cAlbum   = convertCode(gtk_entry_get_text(GTK_ENTRY(widgets.albumEntry)),   pluginConfig.iToUTF8);
    char* cArtist  = convertCode(gtk_entry_get_text(GTK_ENTRY(widgets.artistEntry)),  pluginConfig.iToUTF8);
    char* cTitle   = convertCode(gtk_entry_get_text(GTK_ENTRY(widgets.titleEntry)),   pluginConfig.iToUTF8);
    char* cGenre   = convertCode(gtk_entry_get_text(GTK_ENTRY(widgets.genreEntry)),   pluginConfig.iToUTF8);
    char* cComment = convertCode(gtk_entry_get_text(GTK_ENTRY(widgets.commentEntry)), pluginConfig.iToUTF8);

    const String album   = String(cAlbum,   TagLib::String::UTF8);
    const String artist  = String(cArtist,  TagLib::String::UTF8);
    const String title   = String(cTitle,   TagLib::String::UTF8);
    const String genre   = String(cGenre,   TagLib::String::UTF8);
    const String comment = String(cComment, TagLib::String::UTF8);

    poTag->setAlbum(album);
    poTag->setArtist(artist);
    poTag->setTitle(title);
    poTag->setGenre(genre);
    poTag->setComment(comment);
    poTag->setYear(atoi(gtk_entry_get_text(GTK_ENTRY(widgets.yearEntry))));
    poTag->setTrack(atoi(gtk_entry_get_text(GTK_ENTRY(widgets.trackEntry))));

    free(cAlbum);
    free(cArtist);
    free(cTitle);
    free(cGenre);
    free(cComment);

    oFile.save();
    closeInfoBox(NULL, NULL);
}

static void closeInfoBox(GtkWidget* w, gpointer data)
{
    gtk_widget_destroy(widgets.infoBox);
    widgets.infoBox = NULL;
}

static char* mpcGenerateTitle(const MpcInfo& p_Tags, const char* p_Filename)
{
    TitleInput* input;
    //From titlestring.h
    input            = (TitleInput*) g_malloc0(sizeof(TitleInput));
    input->__size    = XMMS_TITLEINPUT_SIZE;
    input->__version = XMMS_TITLEINPUT_VERSION;
    //end

    char* pFilename = convertCode(p_Filename, pluginConfig.iFromUTF8);

    input->file_name    = g_strdup(g_basename(pFilename));
    input->file_path    = g_dirname(pFilename);
    input->file_ext     = g_strdup("mpc");
    input->date         = g_strdup(p_Tags.date);
    input->track_name   = g_strdup(p_Tags.title);
    input->performer    = g_strdup(p_Tags.artist);
    input->album_name   = g_strdup(p_Tags.album);
    input->track_number = p_Tags.track;
    input->year         = p_Tags.year;
    input->genre        = g_strdup(p_Tags.genre);
    input->comment      = g_strdup(p_Tags.comment);

    char* title = xmms_get_titlestring (xmms_get_gentitle_format(), input);
    if(!title)
        title = g_strdup(input->file_name);
    else if (!*title)
        title = g_strdup(input->file_name);

    free(pFilename);
    free(input->file_name);
    free(input->file_path);
    free(input->file_ext);
    free(input->track_name);
    free(input->performer);
    free(input->album_name);
    free(input->genre);
    free(input->comment);
    free(input->date);
    free(input);
    return title;
}

static void* endThread(FILE* p_Filename, bool release)
{
    if(release)
        lockRelease();
    if(mpcDecoder.isError)
    {
        perror(mpcDecoder.isError);
        free(mpcDecoder.isError);
        mpcDecoder.isError = NULL;
    }
    setAlive(false);
    if(p_Filename)
        fclose(p_Filename);
    if(track.display)
    {
        free(track.display);
        track.display = NULL;
    }
    pthread_exit(NULL);
    return 0;
}

static void* decodeStream(void* data)
{
    lockAcquire();
    const char* filename = static_cast<const char*> (data);
	mpc_reader reader;
	if (mpc_reader_init_stdio(&reader, filename) != MPC_STATUS_OK) {
        mpcDecoder.isError = g_strdup_printf("[xmms-musepack] decodeStream is unable to open %s", filename);
        return endThread(0, true);
	}

	mpc_demux * demux = 0;
	demux = mpc_demux_init(&reader);

	if ( demux == 0 ) {
		mpcDecoder.isError = g_strdup_printf("[xmms-musepack] decodeStream is unable to initialize decoder on %s", filename);
		mpc_reader_exit_stdio(&reader);
		return endThread(0, true);
	}

	mpc_streaminfo info;
	mpc_demux_get_info(demux, &info);
    MpcInfo tags     = getTags(filename);
    track.display    = mpcGenerateTitle(tags, filename);
	track.length     = static_cast<int> (1000 * mpc_streaminfo_get_length(&info));
	track.bitrate    = static_cast<int> (info.average_bitrate);
	track.sampleFreq = info.sample_freq;
	track.channels   = info.channels;
    freeTags(tags);

    MpcPlugin.set_info(track.display, track.length, track.bitrate, track.sampleFreq, track.channels);

// 	mpc_decoder_set_seeking(decoder, &info, pluginConfig.fastSeek);
    setReplaygain(demux);

    MPC_SAMPLE_FORMAT sampleBuffer[MPC_DECODER_BUFFER_LENGTH];
    char xmmsBuffer[MPC_DECODER_BUFFER_LENGTH * 4];

    if (!MpcPlugin.output->open_audio(FMT_S16_LE, track.sampleFreq, track.channels))
    {
		mpcDecoder.isError = g_strdup_printf("[xmms-musepack] decodeStream is unable to open an audio output");
		mpc_reader_exit_stdio(&reader);
        return endThread(0, true);
    }
    else
    {
        mpcDecoder.isOutput = true;
    }

    lockRelease();

	int counter = 2 * track.sampleFreq / 3;
	int status = 0;
    while (isAlive())
    {
        if (getOffset() != -1)
        {
			mpc_demux_seek_second(demux, mpcDecoder.offset);
            setOffset(-1);
			status = 0;
        }

        lockAcquire();
        short iPlaying = MpcPlugin.output->buffer_playing()? 1 : 0;
        int iFree = MpcPlugin.output->buffer_free();
        if (!mpcDecoder.isPause &&  iFree >= ((1152 * 4) << iPlaying) && status != -1)
        {
			status = processBuffer(sampleBuffer, xmmsBuffer, *demux);
			lockRelease();

			if(pluginConfig.dynamicBitrate) {
				counter -= status;
				if(counter < 0) {
					MpcPlugin.set_info(track.display, track.length, track.bitrate, track.sampleFreq, track.channels);
					counter = 2 * track.sampleFreq / 3;
				}
			}
        }
        else
        {
			lockRelease();
			if (mpcDecoder.isPause == FALSE && status == -1 &&
						 MpcPlugin.output->buffer_playing() == FALSE)
				break;
            xmms_usleep(100000);
		}
	}
	mpc_demux_exit(demux);
	mpc_reader_exit_stdio(&reader);
    return endThread(0, false);
}

static int processBuffer(MPC_SAMPLE_FORMAT* sampleBuffer, char* xmmsBuffer, mpc_demux& demux)
{
	mpc_frame_info info;

	info.buffer = sampleBuffer;
	mpc_demux_decode(&demux, &info);

	if (info.bits == -1) return -1; // end of stream

    copyBuffer(sampleBuffer, xmmsBuffer, info.samples);

    if (pluginConfig.dynamicBitrate)
    {
        track.bitrate = static_cast<int> (info.bits * track.sampleFreq / 1152);
    }

    if (pluginConfig.isEq)
    {
		iir(xmmsBuffer, 4 * info.samples);
    }

	MpcPlugin.add_vis_pcm(MpcPlugin.output->written_time(), FMT_S16_LE, track.channels, info.samples * 4, xmmsBuffer);
	MpcPlugin.output->write_audio(xmmsBuffer, info.samples * 4);
	return info.samples;
}

static void setReplaygain(mpc_demux * d)
{
	setReplayLevel(d, 64.82, pluginConfig.replaygain, !pluginConfig.albumGain,
				   pluginConfig.clipPrevention);
}

inline static void lockAcquire()
{
    pthread_mutex_lock(&threadMutex);
}

inline static void lockRelease()
{
    pthread_mutex_unlock(&threadMutex);
}

inline static bool isAlive()
{
    lockAcquire();
    bool isAlive = mpcDecoder.isAlive;
    lockRelease();
    return isAlive;
}

inline static bool isPause()
{
    lockAcquire();
    bool isPause = mpcDecoder.isPause;
    lockRelease();
    return isPause;
}

inline static void setAlive(bool isAlive)
{
    lockAcquire();
    mpcDecoder.isAlive = isAlive;
    lockRelease();
}

inline static double getOffset()
{
    lockAcquire();
    double offset = mpcDecoder.offset;
    lockRelease();
    return offset;
}

inline static void setOffset(double offset)
{
    lockAcquire();
    mpcDecoder.offset = offset;
    lockRelease();
}
