/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/// \file mpc_decoder.c
/// Core decoding routines and logic.

#include <mpcdec/mpcdec.h>
#include <mpcdec/internal.h>
#include <mpcdec/requant.h>
#include <mpcdec/huffman.h>

//SV7 tables
extern const HuffmanTyp*   mpc_table_HuffQ [2] [8];
extern const HuffmanTyp    mpc_table_HuffHdr  [10];
extern const HuffmanTyp    mpc_table_HuffSCFI [ 4];
extern const HuffmanTyp    mpc_table_HuffDSCF [16];


#ifdef MPC_SUPPORT_SV456
//SV4/5/6 tables
extern const HuffmanTyp*   mpc_table_SampleHuff [18];
extern const HuffmanTyp    mpc_table_SCFI_Bundle   [ 8];
extern const HuffmanTyp    mpc_table_DSCF_Entropie [13];
extern const HuffmanTyp    mpc_table_Region_A [16];
extern const HuffmanTyp    mpc_table_Region_B [ 8];
extern const HuffmanTyp    mpc_table_Region_C [ 4];

#endif

#ifndef MPC_LITTLE_ENDIAN
#define SWAP(X) mpc_swap32(X)
#else
#define SWAP(X) X
#endif

#ifdef SCF_HACK
#define SCF_DIFF(SCF, D) (SCF == -128 ? -128 : SCF + D)
#else
#define SCF_DIFF(SCF, D) SCF + D
#endif

#define LOOKUP(x, e, q) mpc_decoder_make_huffman_lookup ( (q), sizeof(q), (x), (e) )
#define DECODE_DSCF() HUFFMAN_DECODE_FASTEST ( d, mpc_table_HuffDSCF, LUTDSCF, 6 )
#define HUFFMAN_DECODE_FASTEST(d,a,b,c) mpc_decoder_huffman_decode_fastest ( (d), (a), (b), 32-(c) )
#define HUFFMAN_DECODE_FASTERER(d,a,b,c) mpc_decoder_huffman_decode_fasterer ( (d), (a), (b), 32-(c) )

mpc_uint8_t     LUT1_0  [1<< 6];
mpc_uint8_t     LUT1_1  [1<< 9];            //  576 Bytes
mpc_uint8_t     LUT2_0  [1<< 7];
mpc_uint8_t     LUT2_1  [1<<10];            // 1152 Bytes
mpc_uint8_t     LUT3_0  [1<< 4];
mpc_uint8_t     LUT3_1  [1<< 5];            //   48 Bytes
mpc_uint8_t     LUT4_0  [1<< 4];
mpc_uint8_t     LUT4_1  [1<< 5];            //   48 Bytes
mpc_uint8_t     LUT5_0  [1<< 6];
mpc_uint8_t     LUT5_1  [1<< 8];            //  320 Bytes
mpc_uint8_t     LUT6_0  [1<< 7];
mpc_uint8_t     LUT6_1  [1<< 7];            //  256 Bytes
mpc_uint8_t     LUT7_0  [1<< 8];
mpc_uint8_t     LUT7_1  [1<< 8];            //  512 Bytes
mpc_uint8_t     LUTDSCF [1<< 6];            //   64 Bytes = 2976 Bytes

//------------------------------------------------------------------------------
// types
//------------------------------------------------------------------------------
enum
    {
        EQ_TAP = 13,                        // length of FIR filter for EQ
        DELAY = ((EQ_TAP + 1) / 2),         // delay of FIR
        FIR_BANDS = 4,                      // number of subbands to be FIR filtered
        MEMSIZE = MPC_DECODER_MEMSIZE,      // overall buffer size
        MEMSIZE2 = (MEMSIZE/2),             // size of one buffer
        MEMMASK = (MEMSIZE-1)
    };

//------------------------------------------------------------------------------
// forward declarations
//------------------------------------------------------------------------------
void mpc_decoder_read_bitstream_sv6(mpc_decoder *d);
void mpc_decoder_read_bitstream_sv7(mpc_decoder *d, mpc_bool_t isFastSeeking);
void mpc_decoder_update_buffer(mpc_decoder *d);
mpc_bool_t mpc_decoder_seek_sample(mpc_decoder *d, mpc_int64_t destsample);
void mpc_decoder_requantisierung(mpc_decoder *d, const mpc_int32_t Last_Band);
void mpc_decoder_seek_to(mpc_decoder *d, mpc_uint32_t bitPos);
void mpc_decoder_seek_forward(mpc_decoder *d, mpc_uint32_t bits);
mpc_uint32_t mpc_decoder_jump_frame(mpc_decoder *d);
void mpc_decoder_fill_buffer(mpc_decoder *d);
void mpc_decoder_reset_state(mpc_decoder *d);
static mpc_uint32_t get_initial_fpos(mpc_decoder *d, mpc_uint32_t StreamVersion);
static __inline mpc_int32_t mpc_decoder_huffman_decode_fastest(mpc_decoder *d, const HuffmanTyp* Table, const mpc_uint8_t* tab, mpc_uint16_t unused_bits);
static void mpc_move_next(mpc_decoder *d);

//------------------------------------------------------------------------------
// utility functions
//------------------------------------------------------------------------------
static mpc_int32_t f_read(mpc_decoder *d, void *ptr, size_t size) 
{ 
    return d->r->read(d->r->data, ptr, size); 
};

static mpc_bool_t f_seek(mpc_decoder *d, mpc_int32_t offset) 
{ 
    return d->r->seek(d->r->data, offset); 
};

static mpc_int32_t f_read_dword(mpc_decoder *d, mpc_uint32_t * ptr, mpc_uint32_t count) 
{
    count = f_read(d, ptr, count << 2) >> 2;
#ifndef MPC_LITTLE_ENDIAN
    {
        mpc_uint32_t n;
        for(n = 0; n< count; n++)
            ptr[n] = mpc_swap32(ptr[n]);
    }
#endif
    return count;
}

//------------------------------------------------------------------------------
// huffman & bitstream functions
//------------------------------------------------------------------------------
static const mpc_uint32_t mask [33] = {
    0x00000000, 0x00000001, 0x00000003, 0x00000007,
    0x0000000F, 0x0000001F, 0x0000003F, 0x0000007F,
    0x000000FF, 0x000001FF, 0x000003FF, 0x000007FF,
    0x00000FFF, 0x00001FFF, 0x00003FFF, 0x00007FFF,
    0x0000FFFF, 0x0001FFFF, 0x0003FFFF, 0x0007FFFF,
    0x000FFFFF, 0x001FFFFF, 0x003FFFFF, 0x007FFFFF,
    0x00FFFFFF, 0x01FFFFFF, 0x03FFFFFF, 0x07FFFFFF,
    0x0FFFFFFF, 0x1FFFFFFF, 0x3FFFFFFF, 0x7FFFFFFF,
    0xFFFFFFFF
};

/* F U N C T I O N S */

// resets bitstream decoding
static void
mpc_decoder_reset_bitstream_decode(mpc_decoder *d) 
{
    d->dword = 0;
    d->pos = 0;
    d->next = 0;
    d->Zaehler = 0;
    d->WordsRead = 0;
}

// reports the number of read bits
static mpc_uint32_t
mpc_decoder_bits_read(mpc_decoder *d) 
{
    return 32 * d->WordsRead + d->pos;
}

static void
mpc_move_next(mpc_decoder *d)
{
    d->Zaehler = (d->Zaehler + 1) & MEMMASK;
    d->dword = d->next;
    d->next = SWAP(d->Speicher[(d->Zaehler + 1) & MEMMASK]);
    d->pos -= 32;
    ++(d->WordsRead);
}

// read desired number of bits out of the bitstream
static __inline mpc_uint32_t
mpc_decoder_bitstream_read(mpc_decoder *d, const mpc_uint32_t bits) 
{
    mpc_uint32_t out = d->dword;

    d->pos += bits;

    if (d->pos < 32) {
        out >>= (32 - d->pos);
    }
    else {
        mpc_move_next(d);
        if (d->pos) {
            out <<= d->pos;
            out |= d->dword >> (32 - d->pos);
        }
    }

    return out & mask[bits];
}

static void
mpc_decoder_make_huffman_lookup(
    mpc_uint8_t* lookup, size_t length, const HuffmanTyp* Table, size_t elements)
{
    size_t i, idx  = elements;
    mpc_uint32_t dval = (mpc_uint32_t)0x80000000L / length * 2;
    mpc_uint32_t val  = dval - 1;

    for ( i = 0; i < length; i++, val += dval ) {
        while ( idx > 0  &&  val >= Table[idx-1].Code )
            idx--;
        *lookup++ = (mpc_uint8_t)idx;
    }

    return;
}

// decode SCFI-bundle (sv4,5,6)
static void
mpc_decoder_scfi_bundle_read(
    mpc_decoder *d,
    const HuffmanTyp* Table, mpc_int8_t* SCFI, mpc_bool_t* DSCF) 
{
    // load preview and decode
    mpc_uint32_t code  = d->dword << d->pos;
    if (d->pos > 26) {
        code |= d->next >> (32 - d->pos);
    }
    while (code < Table->Code) {
        Table++;
    }

    // set the new position within bitstream without performing a dummy-read
    if ((d->pos += Table->Length) >= 32) {
         mpc_move_next(d);
    }

    *SCFI = Table->Value >> 1;
    *DSCF = Table->Value &  1;
}

// basic huffman decoding routine
// works with maximum lengths up to 14
static mpc_int32_t
mpc_decoder_huffman_decode(mpc_decoder *d, const HuffmanTyp *Table) 
{
    // load preview and decode
    mpc_uint32_t code = d->dword << d->pos;
    if (d->pos > 18) {
        code |= d->next >> (32 - d->pos);
    }
    while (code < Table->Code) {
        Table++;
    }

    // set the new position within bitstream without performing a dummy-read
    if ((d->pos += Table->Length) >= 32) {
        mpc_move_next(d);
    }

    return Table->Value;
}

// faster huffman through previewing less bits
// works with maximum lengths up to 10
static mpc_int32_t
mpc_decoder_huffman_decode_fast(mpc_decoder *d, const HuffmanTyp* Table)
{
    // load preview and decode
    mpc_uint32_t code  = d->dword << d->pos;
    if (d->pos > 22) {
        code |= d->next >> (32 - d->pos);
    }
    while (code < Table->Code) {
        Table++;
    }

    // set the new position within bitstream without performing a dummy-read
    if ((d->pos += Table->Length) >= 32) {
        mpc_move_next(d);
    }

    return Table->Value;
}

// even faster huffman through previewing even less bits
// works with maximum lengths up to 5
static mpc_int32_t
mpc_decoder_huffman_decode_faster(mpc_decoder *d, const HuffmanTyp* Table)
{
    // load preview and decode
    mpc_uint32_t code  = d->dword << d->pos;
    if (d->pos > 27) {
        code |= d->next >> (32 - d->pos);
    }
    while (code < Table->Code) {
        Table++;
    }

    // set the new position within bitstream without performing a dummy-read
    if ((d->pos += Table->Length) >= 32) {
        mpc_move_next(d);
    }

    return Table->Value;
}

/* partial lookup table decode */
static mpc_int32_t
mpc_decoder_huffman_decode_fasterer(
    mpc_decoder *d, 
    const HuffmanTyp* Table, const mpc_uint8_t* tab, mpc_uint16_t unused_bits)
{
    // load preview and decode
    mpc_uint32_t code  = d->dword << d->pos;

    if (d->pos > 18) { // preview 14 bits
        code |= d->next >> (32 - d->pos);
    }

    Table += tab [(size_t)(code >> unused_bits) ];

    while (code < Table->Code) {
        Table++;
    }

    // set the new position within bitstream without performing a dummy-read
    if ((d->pos += Table->Length) >= 32) {
        mpc_move_next(d);
    }

    return Table->Value;
}

/* full decode using lookup table */
static __inline mpc_int32_t
mpc_decoder_huffman_decode_fastest(
    mpc_decoder *d, 
    const HuffmanTyp* Table, const mpc_uint8_t* tab, mpc_uint16_t unused_bits)
{
    // load preview and decode
    mpc_uint32_t code  = d->dword << d->pos;

    if (d->pos > unused_bits) {
        code |= d->next >> (32 - d->pos);
    }

    Table+=tab [(size_t)(code >> unused_bits) ];

    // set the new position within bitstream without performing a dummy-read
    if ((d->pos += Table->Length) >= 32) {
        mpc_move_next(d);
    }

    return Table->Value;
}

static void
mpc_decoder_reset_v(mpc_decoder *d) 
{
    memset(d->V_L, 0, sizeof d->V_L);
    memset(d->V_R, 0, sizeof d->V_R);
}

static void
mpc_decoder_reset_synthesis(mpc_decoder *d) 
{
    mpc_decoder_reset_v(d);
}

static void
mpc_decoder_reset_y(mpc_decoder *d) 
{
    memset(d->Y_L, 0, sizeof d->Y_L);
    memset(d->Y_R, 0, sizeof d->Y_R);
}

static void
mpc_decoder_reset_globals(mpc_decoder *d) 
{
    mpc_decoder_reset_bitstream_decode(d);

    d->DecodedFrames  = 0;
    d->SeekTableIndex = 0;
    d->MaxDecodedFrames = 0;
    d->StreamVersion  = 0;
    d->MS_used        = 0;

    memset(d->Y_L          , 0, sizeof d->Y_L           );
    memset(d->Y_R          , 0, sizeof d->Y_R           );
    memset(d->SCF_Index_L     , 0, sizeof d->SCF_Index_L      );
    memset(d->SCF_Index_R     , 0, sizeof d->SCF_Index_R      );
    memset(d->Res_L           , 0, sizeof d->Res_L            );
    memset(d->Res_R           , 0, sizeof d->Res_R            );
    memset(d->SCFI_L          , 0, sizeof d->SCFI_L           );
    memset(d->SCFI_R          , 0, sizeof d->SCFI_R           );
#ifdef MPC_SUPPORT_SV456
    memset(d->DSCF_Flag_L     , 0, sizeof d->DSCF_Flag_L      );
    memset(d->DSCF_Flag_R     , 0, sizeof d->DSCF_Flag_R      );
#endif
    //memset(d->DSCF_Reference_L, 0, sizeof d->DSCF_Reference_L );
    //memset(d->DSCF_Reference_R, 0, sizeof d->DSCF_Reference_R );
    memset(d->Q               , 0, sizeof d->Q                );
    memset(d->MS_Flag         , 0, sizeof d->MS_Flag          );
}

mpc_uint32_t
mpc_decoder_decode_frame(mpc_decoder *d, mpc_uint32_t *in_buffer,
                         mpc_uint32_t in_len, MPC_SAMPLE_FORMAT *out_buffer)
{
  mpc_decoder_reset_bitstream_decode(d);
  if (in_len > sizeof(d->Speicher)) in_len = sizeof(d->Speicher);
  memcpy(d->Speicher, in_buffer, in_len);
   d->dword = SWAP(d->Speicher[0]);
   d->next  = SWAP(d->Speicher[1]);
  switch (d->StreamVersion) {
#ifdef MPC_SUPPORT_SV456
    case 0x04:
    case 0x05:
    case 0x06:
        mpc_decoder_read_bitstream_sv6(d);
        break;
#endif
    case 0x07:
    case 0x17:
        mpc_decoder_read_bitstream_sv7(d, FALSE);
        break;
    default:
        return (mpc_uint32_t)(-1);
  }
  mpc_decoder_requantisierung(d, d->Max_Band);
  mpc_decoder_synthese_filter_float(d, out_buffer);
  return mpc_decoder_bits_read(d);
}

static mpc_uint32_t
mpc_decoder_decode_internal(mpc_decoder *d, MPC_SAMPLE_FORMAT *buffer) 
{
    mpc_uint32_t output_frame_length = MPC_FRAME_LENGTH;

    mpc_uint32_t  FrameBitCnt = 0;

    if (d->DecodedFrames >= d->OverallFrames) {
        return (mpc_uint32_t)(-1);                           // end of file -> abort decoding
    }

    if (d->DecodedFrames == 0 && d->Use_SeekTable)
        d->SeekTable[0] = mpc_decoder_bits_read(d);

    // read jump-info for validity check of frame
    d->FwdJumpInfo  = mpc_decoder_bitstream_read(d, 20);

    d->ActDecodePos = (d->Zaehler << 5) + d->pos;

    // decode data and check for validity of frame
    FrameBitCnt = mpc_decoder_bits_read(d);
    switch (d->StreamVersion) {
#ifdef MPC_SUPPORT_SV456
    case 0x04:
    case 0x05:
    case 0x06:
        mpc_decoder_read_bitstream_sv6(d);
        break;
#endif
    case 0x07:
    case 0x17:
        mpc_decoder_read_bitstream_sv7(d, FALSE);
        break;
    default:
        return (mpc_uint32_t)(-1);
    }
    d->FrameWasValid = mpc_decoder_bits_read(d) - FrameBitCnt == d->FwdJumpInfo;

    d->DecodedFrames++;

    if (d->Use_SeekTable) {
        if (d->SeekTable_Step == 1) {
            d->SeekTable [d->DecodedFrames] = d->FwdJumpInfo + 20;
        } else {
            if ((d->DecodedFrames-1) % d->SeekTable_Step == 0) {
                d->SeekTable[d->SeekTableIndex] = d->SeekTableCounter;
                d->SeekTableIndex  += 1;
                d->SeekTableCounter = 0;
            }
            d->SeekTableCounter += d->FwdJumpInfo + 20;
        }
    }

    // synthesize signal
    mpc_decoder_requantisierung(d, d->Max_Band);

    //if ( d->EQ_activated && PluginSettings.EQbyMPC )
    //    perform_EQ ();

    mpc_decoder_synthese_filter_float(d, buffer);

    // cut off first MPC_DECODER_SYNTH_DELAY zero-samples
    if (d->DecodedFrames == d->OverallFrames  && d->StreamVersion >= 6) {        
        // reconstruct exact filelength
        mpc_int32_t  mod_block   = mpc_decoder_bitstream_read(d,  11);
        mpc_int32_t  FilterDecay;

        if (mod_block == 0) {
            // Encoder bugfix
            mod_block = 1152;                    
        }
        FilterDecay = (mod_block + MPC_DECODER_SYNTH_DELAY) % MPC_FRAME_LENGTH;

        // additional FilterDecay samples are needed for decay of synthesis filter
        if (MPC_DECODER_SYNTH_DELAY + mod_block >= MPC_FRAME_LENGTH) {
            if (!d->TrueGaplessPresent) {
                mpc_decoder_reset_y(d);
            } else {
                mpc_decoder_bitstream_read(d, 20);
                mpc_decoder_read_bitstream_sv7(d, FALSE);
                mpc_decoder_requantisierung(d, d->Max_Band);
            }

            mpc_decoder_synthese_filter_float(d, buffer + 2304);

            output_frame_length = MPC_FRAME_LENGTH + FilterDecay;
        }
        else {                              // there are only FilterDecay samples needed for this frame
            output_frame_length = FilterDecay;
        }
    }

    if (d->samples_to_skip) {
        if (output_frame_length < d->samples_to_skip) {
            d->samples_to_skip -= output_frame_length;
            output_frame_length = 0;
        }
        else {
            output_frame_length -= d->samples_to_skip;
            memmove(
                buffer, 
                buffer + d->samples_to_skip * 2, 
                output_frame_length * 2 * sizeof (MPC_SAMPLE_FORMAT));
            d->samples_to_skip = 0;
        }
    }

    return output_frame_length;
}

mpc_uint32_t mpc_decoder_decode(
    mpc_decoder *d,
    MPC_SAMPLE_FORMAT *buffer, 
    mpc_uint32_t *vbr_update_acc, 
    mpc_uint32_t *vbr_update_bits)
{
    for(;;)
    {
        //const mpc_int32_t MaxBrokenFrames = 0; // PluginSettings.MaxBrokenFrames

        mpc_uint32_t RING = d->Zaehler;
        mpc_int32_t vbr_ring = (RING << 5) + d->pos;

        mpc_uint32_t valid_samples = mpc_decoder_decode_internal(d, buffer);

        if (valid_samples == (mpc_uint32_t)(-1) ) {
            return 0;
        }

        /**************** ERROR CONCEALMENT *****************/
        if (d->FrameWasValid == 0 ) {
            // error occurred in bitstream
            return (mpc_uint32_t)(-1);
        } 
        else {
            if (vbr_update_acc && vbr_update_bits) {
                (*vbr_update_acc) ++;
                vbr_ring = (d->Zaehler << 5) + d->pos - vbr_ring;
                if (vbr_ring < 0) {
                    vbr_ring += 524288;
                }
                (*vbr_update_bits) += vbr_ring;
            }

        }
        mpc_decoder_update_buffer(d);

        if (valid_samples > 0) {
            return valid_samples;
        }
    }
}

void
mpc_decoder_requantisierung(mpc_decoder *d, const mpc_int32_t Last_Band) 
{
    mpc_int32_t     Band;
    mpc_int32_t     n;
    MPC_SAMPLE_FORMAT facL;
    MPC_SAMPLE_FORMAT facR;
    MPC_SAMPLE_FORMAT templ;
    MPC_SAMPLE_FORMAT tempr;
    MPC_SAMPLE_FORMAT* YL;
    MPC_SAMPLE_FORMAT* YR;
    mpc_int16_t*    L;
    mpc_int16_t*    R;

#ifdef MPC_FIXED_POINT
#if MPC_FIXED_POINT_FRACTPART == 14
#define MPC_MULTIPLY_SCF(CcVal, SCF_idx) \
    MPC_MULTIPLY_EX(CcVal, d->SCF[SCF_idx], d->SCF_shift[SCF_idx])
#else

#error FIXME, Cc table is in 18.14 format

#endif
#else
#define MPC_MULTIPLY_SCF(CcVal, SCF_idx) \
    MPC_MULTIPLY(CcVal, d->SCF[SCF_idx])
#endif
    // requantization and scaling of subband-samples
    for ( Band = 0; Band <= Last_Band; Band++ ) {   // setting pointers
        YL = d->Y_L[0] + Band;
        YR = d->Y_R[0] + Band;
        L  = d->Q[Band].L;
        R  = d->Q[Band].R;
        /************************** MS-coded **************************/
        if ( d->MS_Flag [Band] ) {
            if ( d->Res_L [Band] ) {
                if ( d->Res_R [Band] ) {    // M!=0, S!=0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][0]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL   = (templ = MPC_MULTIPLY_FLOAT_INT(facL,*L++))+(tempr = MPC_MULTIPLY_FLOAT_INT(facR,*R++));
                        *YR   = templ - tempr;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][1]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL   = (templ = MPC_MULTIPLY_FLOAT_INT(facL,*L++))+(tempr = MPC_MULTIPLY_FLOAT_INT(facR,*R++));
                        *YR   = templ - tempr;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][2]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL   = (templ = MPC_MULTIPLY_FLOAT_INT(facL,*L++))+(tempr = MPC_MULTIPLY_FLOAT_INT(facR,*R++));
                        *YR   = templ - tempr;
                    }
                } else {    // M!=0, S==0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                    }
                }
            } else {
                if (d->Res_R[Band])    // M==0, S!=0
                {
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YR = - (*YL = MPC_MULTIPLY_FLOAT_INT(facR,*(R++)));
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YR = - (*YL = MPC_MULTIPLY_FLOAT_INT(facR,*(R++)));
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = - (*YL = MPC_MULTIPLY_FLOAT_INT(facR,*(R++)));
                    }
                } else {    // M==0, S==0
                    for ( n = 0; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = 0;
                    }
                }
            }
        }
        /************************** LR-coded **************************/
        else {
            if ( d->Res_L [Band] ) {
                if ( d->Res_R [Band] ) {    // L!=0, R!=0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][0]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][0]);
                    for (n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][1]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][1]);
                    for (; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][2]);
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][2]);
                    for (; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                } else {     // L!=0, R==0
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = 0;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = 0;
                    }
                    facL = MPC_MULTIPLY_SCF( Cc[d->Res_L[Band]] , d->SCF_Index_L[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL = MPC_MULTIPLY_FLOAT_INT(facL,*L++);
                        *YR = 0;
                    }
                }
            }
            else {
                if ( d->Res_R [Band] ) {    // L==0, R!=0
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][0]);
                    for ( n = 0; n < 12; n++, YL += 32, YR += 32 ) {
                        *YL = 0;
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][1]);
                    for ( ; n < 24; n++, YL += 32, YR += 32 ) {
                        *YL = 0;
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                    facR = MPC_MULTIPLY_SCF( Cc[d->Res_R[Band]] , d->SCF_Index_R[Band][2]);
                    for ( ; n < 36; n++, YL += 32, YR += 32 ) {
                        *YL = 0;
                        *YR = MPC_MULTIPLY_FLOAT_INT(facR,*R++);
                    }
                } else {    // L==0, R==0
                    for ( n = 0; n < 36; n++, YL += 32, YR += 32 ) {
                        *YR = *YL = 0;
                    }
                }
            }
        }
    }
}

#ifdef MPC_SUPPORT_SV456
static const mpc_uint8_t Q_res[32][16] = {
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,17},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,3,4,5,6,17,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
{0,1,2,17,0,0,0,0,0,0,0,0,0,0,0,0},
};

/****************************************** SV 6 ******************************************/
void
mpc_decoder_read_bitstream_sv6(mpc_decoder *d) 
{
    mpc_int32_t n,k;
    mpc_int32_t Max_used_Band=0;
    const HuffmanTyp *Table;
    const HuffmanTyp *x1;
    const HuffmanTyp *x2;
    mpc_int8_t  *L;
    mpc_int8_t  *R;
    mpc_int16_t *QL;
    mpc_int16_t *QR;
    mpc_int8_t *ResL = d->Res_L;
    mpc_int8_t *ResR = d->Res_R;

    /************************ HEADER **************************/
    ResL = d->Res_L;
    ResR = d->Res_R;
    for (n=0; n <= d->Max_Band; ++n, ++ResL, ++ResR)
    {
        if      (n<11)           Table = mpc_table_Region_A;
        else if (n>=11 && n<=22) Table = mpc_table_Region_B;
        else /*if (n>=23)*/      Table = mpc_table_Region_C;

        *ResL = Q_res[n][mpc_decoder_huffman_decode(d, Table)];
        if (d->MS_used) {
            d->MS_Flag[n] = (mpc_bool_t) mpc_decoder_bitstream_read(d,  1);
        }
        *ResR = Q_res[n][mpc_decoder_huffman_decode(d, Table)];

        // only perform the following procedure up to the maximum non-zero subband
        if (*ResL || *ResR) Max_used_Band = n;
    }

    /************************* SCFI-Bundle *****************************/
    ResL = d->Res_L;
    ResR = d->Res_R;
    for (n=0; n<=Max_used_Band; ++n, ++ResL, ++ResR) {
        if (*ResL) mpc_decoder_scfi_bundle_read(d, mpc_table_SCFI_Bundle, &(d->SCFI_L[n]), &(d->DSCF_Flag_L[n]));
        if (*ResR) mpc_decoder_scfi_bundle_read(d, mpc_table_SCFI_Bundle, &(d->SCFI_R[n]), &(d->DSCF_Flag_R[n]));
    }

    /***************************** SCFI ********************************/
    ResL = d->Res_L;
    ResR = d->Res_R;
    L    = d->SCF_Index_L[0];
    R    = d->SCF_Index_R[0];
    for (n=0; n <= Max_used_Band; ++n, ++ResL, ++ResR, L+=3, R+=3)
    {
        if (*ResL)
        {
            /*********** DSCF ************/
            if (d->DSCF_Flag_L[n]==1)
            {
                //L[2] = d->DSCF_Reference_L[n];
                switch (d->SCFI_L[n])
                {
                case 3:
                    L[0] = L[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    L[1] = L[0];
                    L[2] = L[1];
                    break;
                case 1:
                    L[0] = L[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    L[1] = L[0] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    L[2] = L[1];
                    break;
                case 2:
                    L[0] = L[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    L[1] = L[0];
                    L[2] = L[1] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    break;
                case 0:
                    L[0] = L[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    L[1] = L[0] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    L[2] = L[1] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    break;
                default:
                    return;
                    break;
                }
            }
            /************ SCF ************/
            else
            {
                switch (d->SCFI_L[n])
                {
                case 3:
                    L[0] = mpc_decoder_bitstream_read(d,  6);
                    L[1] = L[0];
                    L[2] = L[1];
                    break;
                case 1:
                    L[0] = mpc_decoder_bitstream_read(d,  6);
                    L[1] = mpc_decoder_bitstream_read(d,  6);
                    L[2] = L[1];
                    break;
                case 2:
                    L[0] = mpc_decoder_bitstream_read(d,  6);
                    L[1] = L[0];
                    L[2] = mpc_decoder_bitstream_read(d,  6);
                    break;
                case 0:
                    L[0] = mpc_decoder_bitstream_read(d,  6);
                    L[1] = mpc_decoder_bitstream_read(d,  6);
                    L[2] = mpc_decoder_bitstream_read(d,  6);
                    break;
                default:
                    return;
                    break;
                }
            }
            // update Reference for DSCF
            //d->DSCF_Reference_L[n] = L[2];
        }
        if (*ResR)
        {
            //R[2] = d->DSCF_Reference_R[n];
            /*********** DSCF ************/
            if (d->DSCF_Flag_R[n]==1)
            {
                switch (d->SCFI_R[n])
                {
                case 3:
                    R[0] = R[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    R[1] = R[0];
                    R[2] = R[1];
                    break;
                case 1:
                    R[0] = R[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    R[1] = R[0] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    R[2] = R[1];
                    break;
                case 2:
                    R[0] = R[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    R[1] = R[0];
                    R[2] = R[1] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    break;
                case 0:
                    R[0] = R[2] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    R[1] = R[0] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    R[2] = R[1] + mpc_decoder_huffman_decode_fast(d,  mpc_table_DSCF_Entropie);
                    break;
                default:
                    return;
                    break;
                }
            }
            /************ SCF ************/
            else
            {
                switch (d->SCFI_R[n])
                {
                case 3:
                    R[0] = mpc_decoder_bitstream_read(d, 6);
                    R[1] = R[0];
                    R[2] = R[1];
                    break;
                case 1:
                    R[0] = mpc_decoder_bitstream_read(d, 6);
                    R[1] = mpc_decoder_bitstream_read(d, 6);
                    R[2] = R[1];
                    break;
                case 2:
                    R[0] = mpc_decoder_bitstream_read(d, 6);
                    R[1] = R[0];
                    R[2] = mpc_decoder_bitstream_read(d, 6);
                    break;
                case 0:
                    R[0] = mpc_decoder_bitstream_read(d, 6);
                    R[1] = mpc_decoder_bitstream_read(d, 6);
                    R[2] = mpc_decoder_bitstream_read(d, 6);
                    break;
                default:
                    return;
                    break;
                }
            }
            // update Reference for DSCF
            //d->DSCF_Reference_R[n] = R[2];
        }
    }

    /**************************** Samples ****************************/
    ResL = d->Res_L;
    ResR = d->Res_R;
    for (n=0; n <= Max_used_Band; ++n, ++ResL, ++ResR)
    {
        // setting pointers
        x1 = mpc_table_SampleHuff[*ResL];
        x2 = mpc_table_SampleHuff[*ResR];
        QL = d->Q[n].L;
        QR = d->Q[n].R;

        if (x1!=NULL || x2!=NULL)
            for (k=0; k<36; ++k)
            {
                if (x1 != NULL) *QL++ = mpc_decoder_huffman_decode_fast(d,  x1);
                if (x2 != NULL) *QR++ = mpc_decoder_huffman_decode_fast(d,  x2);
            }

        if (*ResL>7 || *ResR>7)
            for (k=0; k<36; ++k)
            {
                if (*ResL>7) *QL++ = (mpc_int16_t)mpc_decoder_bitstream_read(d,  Res_bit[*ResL]) - Dc[*ResL];
                if (*ResR>7) *QR++ = (mpc_int16_t)mpc_decoder_bitstream_read(d,  Res_bit[*ResR]) - Dc[*ResR];
            }
    }
}
#endif //MPC_SUPPORT_SV456
/****************************************** SV 7 ******************************************/
void
mpc_decoder_read_bitstream_sv7(mpc_decoder *d, mpc_bool_t isFastSeeking) 
{
    // these arrays hold decoding results for bundled quantizers (3- and 5-step)
    /*static*/ mpc_int32_t idx30[] = { -1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1,-1, 0, 1};
    /*static*/ mpc_int32_t idx31[] = { -1,-1,-1, 0, 0, 0, 1, 1, 1,-1,-1,-1, 0, 0, 0, 1, 1, 1,-1,-1,-1, 0, 0, 0, 1, 1, 1};
    /*static*/ mpc_int32_t idx32[] = { -1,-1,-1,-1,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1};
    /*static*/ mpc_int32_t idx50[] = { -2,-1, 0, 1, 2,-2,-1, 0, 1, 2,-2,-1, 0, 1, 2,-2,-1, 0, 1, 2,-2,-1, 0, 1, 2};
    /*static*/ mpc_int32_t idx51[] = { -2,-2,-2,-2,-2,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2};

    mpc_int32_t n, k;
    mpc_int32_t Max_used_Band=0;
    const HuffmanTyp *Table;
    mpc_int32_t  idx;
    mpc_int8_t   *L, *R;
    mpc_int16_t  *LQ, *RQ;
    mpc_int8_t   *ResL, *ResR;
    mpc_uint32_t tmp;
    mpc_uint8_t *LUT;
    mpc_uint8_t max_length;

    /***************************** Header *****************************/
    ResL  = d->Res_L;
    ResR  = d->Res_R;

    // first subband
    *ResL = mpc_decoder_bitstream_read(d, 4);
    *ResR = mpc_decoder_bitstream_read(d, 4);
    if (d->MS_used && !(*ResL==0 && *ResR==0)) {
        d->MS_Flag[0] = mpc_decoder_bitstream_read(d, 1);
    } else {
        d->MS_Flag[0] = 0;
    }

    // consecutive subbands
    ++ResL; ++ResR; // increase pointers
    for (n=1; n <= d->Max_Band; ++n, ++ResL, ++ResR)
    {
        idx   = mpc_decoder_huffman_decode_fast(d, mpc_table_HuffHdr);
        *ResL = (idx!=4) ? *(ResL-1) + idx : (int) mpc_decoder_bitstream_read(d, 4);

        idx   = mpc_decoder_huffman_decode_fast(d, mpc_table_HuffHdr);
        *ResR = (idx!=4) ? *(ResR-1) + idx : (int) mpc_decoder_bitstream_read(d, 4);

        if (d->MS_used && !(*ResL==0 && *ResR==0)) {
            d->MS_Flag[n] = mpc_decoder_bitstream_read(d, 1);
        } else {
            d->MS_Flag[n] = 0;
        }

        // only perform following procedures up to the maximum non-zero subband
        if (*ResL!=0 || *ResR!=0) {
            Max_used_Band = n;
        }
    }
    /****************************** SCFI ******************************/
    L     = d->SCFI_L;
    R     = d->SCFI_R;
    ResL  = d->Res_L;
    ResR  = d->Res_R;
    for (n=0; n <= Max_used_Band; ++n, ++L, ++R, ++ResL, ++ResR) {
        if (*ResL) *L = mpc_decoder_huffman_decode_faster(d, mpc_table_HuffSCFI);
        if (*ResR) *R = mpc_decoder_huffman_decode_faster(d, mpc_table_HuffSCFI);
    }

    /**************************** SCF/DSCF ****************************/
    ResL  = d->Res_L;
    ResR  = d->Res_R;
    L     = d->SCF_Index_L[0];
    R     = d->SCF_Index_R[0];
    for (n=0; n<=Max_used_Band; ++n, ++ResL, ++ResR, L+=3, R+=3) {
        if (*ResL)
        {
            //L[2] = d->DSCF_Reference_L[n];
            switch (d->SCFI_L[n])
            {
            case 1:
                idx  = DECODE_DSCF();
                L[0] = (idx!=8) ? SCF_DIFF(L[2], idx) : mpc_decoder_bitstream_read(d, 6);
                idx  = DECODE_DSCF();
                L[1] = (idx!=8) ? SCF_DIFF(L[0], idx) : mpc_decoder_bitstream_read(d, 6);
                L[2] = L[1];
                break;
            case 3:
                idx  = DECODE_DSCF();
                L[0] = (idx!=8) ? SCF_DIFF(L[2], idx) : mpc_decoder_bitstream_read(d, 6);
                L[1] = L[0];
                L[2] = L[1];
                break;
            case 2:
                idx  = DECODE_DSCF();
                L[0] = (idx!=8) ? SCF_DIFF(L[2], idx) : mpc_decoder_bitstream_read(d, 6);
                L[1] = L[0];
                idx  = DECODE_DSCF();
                L[2] = (idx!=8) ? SCF_DIFF(L[1], idx) : mpc_decoder_bitstream_read(d, 6);
                break;
            case 0:
                idx  = DECODE_DSCF();
                L[0] = (idx!=8) ? SCF_DIFF(L[2], idx) : mpc_decoder_bitstream_read(d, 6);
                idx  = DECODE_DSCF();
                L[1] = (idx!=8) ? SCF_DIFF(L[0], idx) : mpc_decoder_bitstream_read(d, 6);
                idx  = DECODE_DSCF();
                L[2] = (idx!=8) ? SCF_DIFF(L[1], idx) : mpc_decoder_bitstream_read(d, 6);
                break;
            default:
                return;
                break;
            }
            // update Reference for DSCF
            //d->DSCF_Reference_L[n] = L[2];
        }
        if (*ResR)
        {
            //R[2] = d->DSCF_Reference_R[n];
            switch (d->SCFI_R[n])
            {
            case 1:
                idx  = DECODE_DSCF();
                R[0] = (idx!=8) ? SCF_DIFF(R[2], idx) : mpc_decoder_bitstream_read(d, 6);
                idx  = DECODE_DSCF();
                R[1] = (idx!=8) ? SCF_DIFF(R[0], idx) : mpc_decoder_bitstream_read(d, 6);
                R[2] = R[1];
                break;
            case 3:
                idx  = DECODE_DSCF();
                R[0] = (idx!=8) ? SCF_DIFF(R[2], idx) : mpc_decoder_bitstream_read(d, 6);
                R[1] = R[0];
                R[2] = R[1];
                break;
            case 2:
                idx  = DECODE_DSCF();
                R[0] = (idx!=8) ? SCF_DIFF(R[2], idx) : mpc_decoder_bitstream_read(d, 6);
                R[1] = R[0];
                idx  = DECODE_DSCF();
                R[2] = (idx!=8) ? SCF_DIFF(R[1], idx) : mpc_decoder_bitstream_read(d, 6);
                break;
            case 0:
                idx  = DECODE_DSCF();
                R[0] = (idx!=8) ? SCF_DIFF(R[2], idx) : mpc_decoder_bitstream_read(d, 6);
                idx  = DECODE_DSCF();
                R[1] = (idx!=8) ? SCF_DIFF(R[0], idx) : mpc_decoder_bitstream_read(d, 6);
                idx  = DECODE_DSCF();
                R[2] = (idx!=8) ? SCF_DIFF(R[1], idx) : mpc_decoder_bitstream_read(d, 6);
                break;
            default:
                return;
                break;
            }
            // update Reference for DSCF
            //d->DSCF_Reference_R[n] = R[2];
        }
    }

    if(isFastSeeking)
        return;

    /***************************** Samples ****************************/
    ResL = d->Res_L;
    ResR = d->Res_R;
    LQ   = d->Q[0].L;
    RQ   = d->Q[0].R;
    for (n=0; n <= Max_used_Band; ++n, ++ResL, ++ResR, LQ+=36, RQ+=36)
    {
        /************** links **************/
        switch (*ResL)
        {
        case  -2: case  -3: case  -4: case  -5: case  -6: case  -7: case  -8: case  -9:
        case -10: case -11: case -12: case -13: case -14: case -15: case -16: case -17:
            LQ += 36;
            break;
        case -1:
            for (k=0; k<36; k++ ) {
                tmp  = mpc_random_int(d);
                *LQ++ = ((tmp >> 24) & 0xFF) + ((tmp >> 16) & 0xFF) + ((tmp >>  8) & 0xFF) + ((tmp >>  0) & 0xFF) - 510;
            }
            break;
        case 0:
            LQ += 36;// increase pointer
            break;
        case 1:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][1];
                LUT = LUT1_1;
                max_length = 9;
            } else {
                Table = mpc_table_HuffQ[0][1];
                LUT = LUT1_0;
                max_length = 6;
            }
            for (k=0; k<12; ++k)
            {
                idx   = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                *LQ++ = idx30[idx];
                *LQ++ = idx31[idx];
                *LQ++ = idx32[idx];
            }
            break;
        case 2:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][2];
                LUT = LUT2_1;
                max_length = 10;
            } else {
                Table = mpc_table_HuffQ[0][2];
                LUT = LUT2_0;
                max_length = 7;
            }
            for (k=0; k<18; ++k)
            {
                idx   = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                *LQ++ = idx50[idx];
                *LQ++ = idx51[idx];
            }
            break;
        case 3:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][3];
                LUT = LUT3_1;
                max_length = 5;
            } else {
                Table = mpc_table_HuffQ[0][3];
                LUT = LUT3_0;
                max_length = 4;
            }
            for (k=0; k<36; ++k)
                *LQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
            break;
        case 4:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][4];
                LUT = LUT4_1;
                max_length = 5;
            } else {
                Table = mpc_table_HuffQ[0][4];
                LUT = LUT4_0;
                max_length = 4;
            }
            for (k=0; k<36; ++k)
                *LQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
            break;
        case 5:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][5];
                LUT = LUT5_1;
                max_length = 8;
            } else {
                Table = mpc_table_HuffQ[0][5];
                LUT = LUT5_0;
                max_length = 6;
            }
            for (k=0; k<36; ++k)
                *LQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
            break;
        case 6:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][6];
                LUT = LUT6_1;
                max_length = 7;
                for (k=0; k<36; ++k)
                    *LQ++ = HUFFMAN_DECODE_FASTERER ( d, Table, LUT, max_length );
            } else {
                Table = mpc_table_HuffQ[0][6];
                LUT = LUT6_0;
                max_length = 7;
                for (k=0; k<36; ++k)
                    *LQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
            }
            break;
        case 7:
            if (mpc_decoder_bitstream_read(d, 1)) {
                Table = mpc_table_HuffQ[1][7];
                LUT = LUT7_1;
                max_length = 8;
                for (k=0; k<36; ++k)
                    *LQ++ = HUFFMAN_DECODE_FASTERER ( d, Table, LUT, max_length );
            } else {
                Table = mpc_table_HuffQ[0][7];
                LUT = LUT7_0;
                max_length = 8;
                for (k=0; k<36; ++k)
                    *LQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
            }
            break;
        case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
            tmp = Dc[*ResL];
            for (k=0; k<36; ++k)
                *LQ++ = (mpc_int16_t) mpc_decoder_bitstream_read(d, Res_bit[*ResL]) - tmp;
            break;
        default:
            return;
        }
        /************** rechts **************/
        switch (*ResR)
        {
        case  -2: case  -3: case  -4: case  -5: case  -6: case  -7: case  -8: case  -9:
        case -10: case -11: case -12: case -13: case -14: case -15: case -16: case -17:
            RQ += 36;
            break;
        case -1:
                for (k=0; k<36; k++ ) {
                    tmp  = mpc_random_int(d);
                    *RQ++ = ((tmp >> 24) & 0xFF) + ((tmp >> 16) & 0xFF) + ((tmp >>  8) & 0xFF) + ((tmp >>  0) & 0xFF) - 510;
                }
                break;
            case 0:
                RQ += 36;// increase pointer
                break;
            case 1:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][1];
                    LUT = LUT1_1;
                    max_length = 9;
                } else {
                    Table = mpc_table_HuffQ[0][1];
                    LUT = LUT1_0;
                    max_length = 6;
                }
                for (k=0; k<12; ++k)
                {
                    idx = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                    *RQ++ = idx30[idx];
                    *RQ++ = idx31[idx];
                    *RQ++ = idx32[idx];
                }
                break;
            case 2:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][2];
                    LUT = LUT2_1;
                    max_length = 10;
                } else {
                    Table = mpc_table_HuffQ[0][2];
                    LUT = LUT2_0;
                    max_length = 7;
                }
                for (k=0; k<18; ++k)
                {
                    idx = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                    *RQ++ = idx50[idx];
                    *RQ++ = idx51[idx];
                }
                break;
            case 3:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][3];
                    LUT = LUT3_1;
                    max_length = 5;
                } else {
                    Table = mpc_table_HuffQ[0][3];
                    LUT = LUT3_0;
                    max_length = 4;
                }
                for (k=0; k<36; ++k)
                    *RQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                break;
            case 4:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][4];
                    LUT = LUT4_1;
                    max_length = 5;
                } else {
                    Table = mpc_table_HuffQ[0][4];
                    LUT = LUT4_0;
                    max_length = 4;
                }
                for (k=0; k<36; ++k)
                    *RQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                break;
            case 5:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][5];
                    LUT = LUT5_1;
                    max_length = 8;
                } else {
                    Table = mpc_table_HuffQ[0][5];
                    LUT = LUT5_0;
                    max_length = 6;
                }
                for (k=0; k<36; ++k)
                    *RQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                break;
            case 6:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][6];
                    LUT = LUT6_1;
                    max_length = 7;
                    for (k=0; k<36; ++k)
                        *RQ++ = HUFFMAN_DECODE_FASTERER ( d, Table, LUT, max_length );
                } else {
                    Table = mpc_table_HuffQ[0][6];
                    LUT = LUT6_0;
                    max_length = 7;
                    for (k=0; k<36; ++k)
                        *RQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                }
                break;
            case 7:
                if (mpc_decoder_bitstream_read(d, 1)) {
                    Table = mpc_table_HuffQ[1][7];
                    LUT = LUT7_1;
                    max_length = 8;
                    for (k=0; k<36; ++k)
                        *RQ++ = HUFFMAN_DECODE_FASTERER ( d, Table, LUT, max_length );
                } else {
                    Table = mpc_table_HuffQ[0][7];
                    LUT = LUT7_0;
                    max_length = 8;
                    for (k=0; k<36; ++k)
                        *RQ++ = HUFFMAN_DECODE_FASTEST ( d, Table, LUT, max_length );
                }
                break;
            case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
                tmp = Dc[*ResR];
                for (k=0; k<36; ++k)
                    *RQ++ = (mpc_int16_t) mpc_decoder_bitstream_read(d, Res_bit[*ResR]) - tmp;
                break;
            default:
                return;
        }
    }
}

void mpc_decoder_setup(mpc_decoder *d, mpc_reader *r)
{
  d->r = r;

  d->MPCHeaderPos = 0;
  d->StreamVersion = 0;
  d->MS_used = 0;
  d->FwdJumpInfo = 0;
  d->ActDecodePos = 0;
  d->FrameWasValid = 0;
  d->OverallFrames = 0;
  d->DecodedFrames = 0;
  d->MaxDecodedFrames = 0;
  d->TrueGaplessPresent = 0;
  d->WordsRead = 0;
  d->Max_Band = 0;
  d->SampleRate = 0;
  d->__r1 = 1;
  d->__r2 = 1;

  d->dword = 0;
  d->pos = 0;
  d->Zaehler = 0;
  d->Ring = 0;
  d->WordsRead = 0;
  d->Max_Band = 0;
  d->SeekTable = NULL;
  d->Use_FastSeek = TRUE;
  d->Use_SeekTable = TRUE;
  d->Use_StaticSeekTable = FALSE;
  d->SeekTable_Step = 1;
  d->SeekTableIndex = 0;
  d->SeekTableCounter = 0;
  d->Max_SeekTable_Size = 0;

  mpc_decoder_initialisiere_quantisierungstabellen(d, 1.0f);
#if 0
    mpc_decoder_init_huffman_sv6(d);
    mpc_decoder_init_huffman_sv7(d);
#endif

  LOOKUP ( mpc_table_HuffQ[0][1], 27, LUT1_0  );
  LOOKUP ( mpc_table_HuffQ[1][1], 27, LUT1_1  );
  LOOKUP ( mpc_table_HuffQ[0][2], 25, LUT2_0  );
  LOOKUP ( mpc_table_HuffQ[1][2], 25, LUT2_1  );
  LOOKUP ( mpc_table_HuffQ[0][3], 7,  LUT3_0  );
  LOOKUP ( mpc_table_HuffQ[1][3], 7,  LUT3_1  );
  LOOKUP ( mpc_table_HuffQ[0][4], 9,  LUT4_0  );
  LOOKUP ( mpc_table_HuffQ[1][4], 9,  LUT4_1  );
  LOOKUP ( mpc_table_HuffQ[0][5], 15, LUT5_0  );
  LOOKUP ( mpc_table_HuffQ[1][5], 15, LUT5_1  );
  LOOKUP ( mpc_table_HuffQ[0][6], 31, LUT6_0  );
  LOOKUP ( mpc_table_HuffQ[1][6], 31, LUT6_1  );
  LOOKUP ( mpc_table_HuffQ[0][7], 63, LUT7_0  );
  LOOKUP ( mpc_table_HuffQ[1][7], 63, LUT7_1  );
  LOOKUP ( mpc_table_HuffDSCF,    16, LUTDSCF );
}

void mpc_decoder_destroy(mpc_decoder *d)
{
    free(d->Speicher);
    if (d->SeekTable != NULL && d->Use_StaticSeekTable == FALSE)
        free(d->SeekTable);
}

void mpc_decoder_set_streaminfo(mpc_decoder *d, mpc_streaminfo *si)
{
    mpc_uint16_t seekTableSize;

    mpc_decoder_reset_synthesis(d);
    mpc_decoder_reset_globals(d);

    d->StreamVersion      = si->stream_version;
    d->MS_used            = si->ms;
    d->Max_Band           = si->max_band;
    d->OverallFrames      = si->frames;
    d->MPCHeaderPos       = si->header_position;
    d->TrueGaplessPresent = si->is_true_gapless;
    d->SampleRate         = (mpc_int32_t)si->sample_freq;

    d->samples_to_skip = MPC_DECODER_SYNTH_DELAY;

    if (d->SeekTable != NULL && d->Use_StaticSeekTable == FALSE)
        free(d->SeekTable);

    if (d->Use_SeekTable) {
        if (d->Use_StaticSeekTable == FALSE) {
            if (d->Max_SeekTable_Size == 0) {
                seekTableSize = si->frames +1;
            } else {
                seekTableSize = min(si->frames +1, d->Max_SeekTable_Size / sizeof(mpc_uint32_t));
            }
            d->SeekTable = (mpc_uint32_t*) calloc(seekTableSize, sizeof(mpc_uint32_t));
            d->SeekTable_Step = (si->frames +1) / seekTableSize;
            if ((si->frames +1) % seekTableSize)
                d->SeekTable_Step+=1;
        } else {
            seekTableSize = d->Max_SeekTable_Size / sizeof(mpc_uint32_t);
            d->SeekTable_Step = (si->frames +1) / seekTableSize;
            if ((si->frames +1) % seekTableSize)
                d->SeekTable_Step+=1;
        }
    }
}

mpc_bool_t mpc_decoder_initialize(mpc_decoder *d, mpc_streaminfo *si) 
{
    mpc_uint32_t bitPos;
    mpc_uint32_t fpos;

    d->Speicher = malloc(MPC_DECODER_MEMSIZE * sizeof(mpc_uint32_t));
    mpc_decoder_set_streaminfo(d, si);

     // setting position to the beginning of the data-bitstream
     bitPos = get_initial_fpos(d, d->StreamVersion);
     fpos = bitPos >> 5;

    // fill buffer and initialize decoder
  	f_seek(d, fpos*4 + d->MPCHeaderPos);
    f_read_dword(d, d->Speicher, MEMSIZE );
    d->Ring = 0;
    d->Zaehler = 0;
    d->pos = bitPos & 31;
    d->WordsRead = fpos;
    d->dword = SWAP(d->Speicher[0]);
    d->next = SWAP(d->Speicher[1]);
    d->SeekTable_Step = 1;

    return TRUE;
}

//---------------------------------------------------------------
// will seek from the beginning of the file to the desired
// position in ms (given by seek_needed)
//---------------------------------------------------------------
#if 0
static void
helper1(mpc_decoder *d, mpc_uint32_t bitpos) 
{
    f_seek(d, (bitpos >> 5) * 4 + d->MPCHeaderPos);
    f_read_dword(d, d->Speicher, 2);
    d->dword = SWAP(d->Speicher[d->Zaehler = 0]);
    d->pos = bitpos & 31;
}

static void
helper2(mpc_decoder *d, mpc_uint32_t bitpos) 
{
    f_seek(d, (bitpos>>5) * 4 + d->MPCHeaderPos);
    f_read_dword(d, d->Speicher, MEMSIZE);
    d->dword = SWAP(d->Speicher[d->Zaehler = 0]);
    d->pos = bitpos & 31;
}

static void
helper3(mpc_decoder *d, mpc_uint32_t bitpos, mpc_uint32_t* buffoffs) 
{
    d->pos = bitpos & 31;
    bitpos >>= 5;
    if ((mpc_uint32_t)(bitpos - *buffoffs) >= MEMSIZE - 2) {
        *buffoffs = bitpos;
        f_seek(d, bitpos * 4L + d->MPCHeaderPos);
        f_read_dword(d, d->Speicher, MEMSIZE );
    }
    d->dword = SWAP(d->Speicher[d->Zaehler = bitpos - *buffoffs ]);
}
#endif

// jumps over the current frame
mpc_uint32_t mpc_decoder_jump_frame(mpc_decoder *d) {

    mpc_uint32_t frameSize;

    // ensure the buffer is full
    mpc_decoder_update_buffer(d);

    // bits in frame
    frameSize = mpc_decoder_bitstream_read(d, 20);

    // jump forward
    mpc_decoder_seek_forward(d, frameSize);

    return frameSize + 20;

}

static mpc_uint32_t get_initial_fpos(mpc_decoder *d, mpc_uint32_t StreamVersion)
{
    mpc_uint32_t fpos = 0;
    (void) StreamVersion;
    switch ( d->StreamVersion ) {                                                  // setting position to the beginning of the data-bitstream
    case  0x04: fpos =  48; break;
    case  0x05:
    case  0x06: fpos =  64; break;
    case  0x07:
    case  0x17: fpos = 200; break;
    }
    return fpos;
}

mpc_bool_t mpc_decoder_seek_seconds(mpc_decoder *d, double seconds) 
{
    return mpc_decoder_seek_sample(d, (mpc_int64_t)(seconds * (double)d->SampleRate + 0.5));
}

void mpc_decoder_reset_state(mpc_decoder *d) {

    memset(d->Y_L          , 0, sizeof d->Y_L           );
    memset(d->Y_R          , 0, sizeof d->Y_R           );
#ifdef SCF_HACK
    memset(d->SCF_Index_L     , -128, sizeof d->SCF_Index_L      );
    memset(d->SCF_Index_R     , -128, sizeof d->SCF_Index_R      );
#else
    memset(d->SCF_Index_L     , 0, sizeof d->SCF_Index_L      );
    memset(d->SCF_Index_R     , 0, sizeof d->SCF_Index_R      );
#endif
    memset(d->Res_L           , 0, sizeof d->Res_L            );
    memset(d->Res_R           , 0, sizeof d->Res_R            );
    memset(d->SCFI_L          , 0, sizeof d->SCFI_L           );
    memset(d->SCFI_R          , 0, sizeof d->SCFI_R           );
#ifdef MPC_SUPPORT_SV456
    memset(d->DSCF_Flag_L     , 0, sizeof d->DSCF_Flag_L      );
    memset(d->DSCF_Flag_R     , 0, sizeof d->DSCF_Flag_R      );
#endif
    //memset(d->DSCF_Reference_L, 0, sizeof d->DSCF_Reference_L );
    //memset(d->DSCF_Reference_R, 0, sizeof d->DSCF_Reference_R );
    memset(d->Q               , 0, sizeof d->Q                );
    memset(d->MS_Flag         , 0, sizeof d->MS_Flag          );
}

mpc_bool_t mpc_decoder_seek_sample(mpc_decoder *d, mpc_int64_t destsample) 
{
    mpc_uint32_t fpos;
    mpc_uint32_t seekFrame = 0;   // the frame to seek to
    mpc_uint32_t lastFrame = 0;   // last frame to seek to before scanning scale factors
    mpc_uint32_t delta = 0;       // direction of seeks

     destsample += MPC_DECODER_SYNTH_DELAY;
     seekFrame = (mpc_uint32_t) ((destsample) / MPC_FRAME_LENGTH);
     d->samples_to_skip = (mpc_uint32_t)((destsample) % MPC_FRAME_LENGTH);

    // prevent from desired position out of allowed range
    seekFrame = seekFrame < d->OverallFrames  ?  seekFrame  :  d->OverallFrames;

    // seek direction (note: avoids casting to int64)
    delta = (d->DecodedFrames > seekFrame ? -(mpc_int32_t)(d->DecodedFrames - seekFrame) : (mpc_int32_t)(seekFrame - d->DecodedFrames));

    // update max decoded frames
    if (d->DecodedFrames > d->MaxDecodedFrames)
        d->MaxDecodedFrames = d->DecodedFrames;

    if (seekFrame > 33)
        lastFrame = seekFrame - 33 + 1 - d->SeekTable_Step;

    if ((!d->Use_SeekTable && delta < 0) || d->MaxDecodedFrames == 0)
    {

        mpc_decoder_reset_state(d);

        // starts from the beginning since no frames have been decoded yet, or not using seek table
        fpos = get_initial_fpos(d, d->StreamVersion);

        // seek to the first frame
        mpc_decoder_seek_to(d, fpos);

        // reset number of decoded frames
        d->DecodedFrames = 0;

        if (d->Use_SeekTable)
        {
            // jump to the last frame, updating seek table
            if (d->SeekTable_Step == 1)
            {
                d->SeekTable[0] = (mpc_uint32_t)fpos;
                for (;d->DecodedFrames < lastFrame; d->DecodedFrames++)
                    d->SeekTable[d->DecodedFrames+1] = mpc_decoder_jump_frame(d);
            }
            else
            {
                d->SeekTableIndex = 0;
                d->SeekTableCounter = (mpc_uint32_t)fpos;
                for (;d->DecodedFrames < lastFrame; d->DecodedFrames++)
                {
                    if (d->DecodedFrames % d->SeekTable_Step == 0)
                    {
                        d->SeekTable[d->SeekTableIndex] = d->SeekTableCounter;
                        d->SeekTableIndex += 1;
                        d->SeekTableCounter = 0;
                    }
                    d->SeekTableCounter += mpc_decoder_jump_frame(d);
                }
            }
        }
        else
        {
            // just jump to the last frame
            for (;d->DecodedFrames < lastFrame; d->DecodedFrames++)
                mpc_decoder_jump_frame(d);
        }
    } 
    else if (delta < 0)
    {

        mpc_decoder_reset_state(d);

        // jumps backwards using the seek table
        fpos = d->SeekTable[0];
        if (d->SeekTable_Step == 1)
        {
            for (d->DecodedFrames = 0;d->DecodedFrames < lastFrame; d->DecodedFrames++)
            fpos += d->SeekTable[d->DecodedFrames+1];
        }
        else
        {
            d->SeekTableIndex = 0;
            //d->SeekTableCounter = 0;
            for (d->DecodedFrames = 0;d->DecodedFrames < lastFrame; d->DecodedFrames+=d->SeekTable_Step, d->SeekTableIndex++)
                fpos += d->SeekTable[d->SeekTableIndex+1];
            d->SeekTableCounter = d->SeekTable[d->SeekTableIndex];
        }
        mpc_decoder_seek_to(d, fpos);

    }
    else if (delta > 33)
    {

        mpc_decoder_reset_state(d);

        // jumps forward from the current position
        if (d->Use_SeekTable)
        {
            if (d->MaxDecodedFrames > lastFrame) // REVIEW: Correct?? or (d->MaxDecodedFrames > d->DecodedFrames)
            { 
                // jump to the last usable position in the seek table
                if (d->SeekTable_Step == 1)
                {
                    fpos = mpc_decoder_bits_read(d);
                    for (; d->DecodedFrames < d->MaxDecodedFrames && d->DecodedFrames < lastFrame; d->DecodedFrames++)
                        fpos += d->SeekTable[d->DecodedFrames+1];
                } 
                else
                {
                    // could test SeekTable offset and jump to next entry but this is easier for now...
                    //d->SeekTableIndex = 0;
                    //d->SeekTableCounter = 0;
                    fpos = d->SeekTable[0];
                    d->SeekTableIndex = 0;
                    for (d->DecodedFrames = 0;d->DecodedFrames < d->MaxDecodedFrames && d->DecodedFrames < lastFrame; d->DecodedFrames+=d->SeekTable_Step, d->SeekTableIndex++)
                        fpos += d->SeekTable[d->SeekTableIndex+1];
                    d->SeekTableCounter = d->SeekTable[d->SeekTableIndex];
                }
                mpc_decoder_seek_to(d, fpos);
            }

            if (d->SeekTable_Step == 1) 
            {
                for (;d->DecodedFrames < lastFrame; d->DecodedFrames++)
                    d->SeekTable[d->DecodedFrames+1] = mpc_decoder_jump_frame(d);
            }
            else 
            {
                for (;d->DecodedFrames < lastFrame; d->DecodedFrames++)
                {
                    if (d->DecodedFrames % d->SeekTable_Step == 0)
                    {
                        d->SeekTable[d->SeekTableIndex] = d->SeekTableCounter;
                        d->SeekTableIndex += 1;
                        d->SeekTableCounter = 0;
                    }
                    d->SeekTableCounter += mpc_decoder_jump_frame(d);
                }
            }
        } 
        else 
        {
            for (;d->DecodedFrames < lastFrame; d->DecodedFrames++)
                mpc_decoder_jump_frame(d);
        }
    }

    // REVIEW: Needed?
    mpc_decoder_update_buffer(d);

    for (;d->DecodedFrames < seekFrame; d->DecodedFrames++) {
        mpc_uint32_t   FrameBitCnt;
        d->FwdJumpInfo  = mpc_decoder_bitstream_read(d, 20);    // read jump-info
        d->ActDecodePos = (d->Zaehler << 5) + d->pos;
        FrameBitCnt  = mpc_decoder_bits_read(d);
  	    // scanning the scalefactors (and check for validity of frame)
        if (d->StreamVersion >= 7)  {
            mpc_decoder_read_bitstream_sv7(d, d->Use_FastSeek && (d->DecodedFrames < seekFrame - 1));
        }
        else {
#ifdef MPC_SUPPORT_SV456
            mpc_decoder_read_bitstream_sv6(d);
#else
            return FALSE;
#endif
        }
        FrameBitCnt = mpc_decoder_bits_read(d) - FrameBitCnt;

        if (d->Use_FastSeek && d->FwdJumpInfo > FrameBitCnt)
            mpc_decoder_seek_forward(d, d->FwdJumpInfo - FrameBitCnt);
        else if (FrameBitCnt != d->FwdJumpInfo )
            // Bug in perform_jump;
            return FALSE;

        // REVIEW: Only if decodedFrames < maxDecodedFrames??
        if (d->Use_SeekTable)
        {
            if (d->SeekTable_Step == 1)
            {
                // check that the frame length corresponds with any data already in the seek table
                if (d->SeekTable[d->DecodedFrames+1] != 0 && d->SeekTable[d->DecodedFrames+1] != d->FwdJumpInfo + 20)
                    return FALSE;
                d->SeekTable [d->DecodedFrames+1] = d->FwdJumpInfo + 20;
            }
            else
            {
                if (d->DecodedFrames % d->SeekTable_Step == 0)
                {
                    if (d->SeekTable[d->SeekTableIndex] != 0 && d->SeekTable[d->SeekTableIndex] != d->SeekTableCounter)
                        return FALSE;
                    d->SeekTable[d->SeekTableIndex] = d->SeekTableCounter;
                    d->SeekTableIndex += 1;
                    d->SeekTableCounter = 0;
                }
                d->SeekTableCounter += d->FwdJumpInfo + 20;
            }
        }

        // update buffer
        mpc_decoder_update_buffer(d);

        if (d->DecodedFrames == seekFrame - 1)
        {
            // initialize the synth correctly for perfect decoding
            mpc_decoder_requantisierung(d, d->Max_Band);
            mpc_decoder_synthese_filter_float(d, NULL);
        }
    }

    return TRUE;
}

void mpc_decoder_fill_buffer(mpc_decoder *d)
{
    f_read_dword(d, d->Speicher, MEMSIZE);
    d->dword = SWAP(d->Speicher[d->Zaehler = 0]);
    d->next = SWAP(d->Speicher[1]);
    d->Ring = 0;
}

void mpc_decoder_update_buffer(mpc_decoder *d) 
{
    if ((d->Ring ^ d->Zaehler) & MEMSIZE2) {
        // update buffer
        f_read_dword(d, d->Speicher + (d->Ring & MEMSIZE2), MEMSIZE2);
        d->Ring = d->Zaehler;
    }
}


void mpc_decoder_seek_to(mpc_decoder *d, mpc_uint32_t bitPos)
{
    // required dword
    mpc_uint32_t fpos = (bitPos >> 5);
    mpc_uint32_t bufferStart = d->WordsRead - d->Zaehler;
    if ((d->Zaehler & MEMSIZE2) != FALSE)
        bufferStart += MEMSIZE2;

    if (fpos >= bufferStart && fpos < bufferStart + MEMSIZE)
    {
        // required position is within the buffer, no need to seek
        d->Zaehler = (fpos - bufferStart + ((d->Zaehler & MEMSIZE2) != FALSE ? MEMSIZE2 : 0)) & MEMMASK;
        d->pos = bitPos & 31;
        d->WordsRead = fpos;
        d->dword = SWAP(d->Speicher[d->Zaehler]);
        d->next = SWAP(d->Speicher[(d->Zaehler + 1) & MEMMASK]);

        mpc_decoder_update_buffer(d);
    } 
    else
    {
        // DWORD aligned
        f_seek(d, fpos*4 + d->MPCHeaderPos);
        d->Zaehler = 0;
        d->pos = bitPos & 31;
        d->WordsRead = fpos;

        mpc_decoder_fill_buffer(d);
    }
}

void mpc_decoder_seek_forward(mpc_decoder *d, mpc_uint32_t bits)
{  	 
    bits += d->pos;
    d->pos = bits & 31;
    bits = bits >> 5; // to DWORDs
    d->Zaehler = (d->Zaehler + bits) & MEMMASK;
    d->dword = SWAP(d->Speicher[d->Zaehler]);
    d->next = SWAP(d->Speicher[(d->Zaehler + 1) & MEMMASK]);
    d->WordsRead += bits;
}

void mpc_decoder_set_seek_table(mpc_decoder *d, mpc_uint32_t *seek_table, mpc_uint32_t max_table_size)
{
    d->Use_StaticSeekTable = TRUE;
    d->SeekTable = seek_table;
    d->Max_SeekTable_Size = max_table_size;
}
