/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <string.h>
#include <assert.h>
#include "mppdec.h"

// Todo:
// Intercept InputBuff-Buffer Overflows less frequently, but leave some headroom???
// Cleanly number the functions...
// Calculate_New_V: Save at all relevant places at the end
// Make 3DNow!-Code (Assembler instructions) more independent from each other (reorder)
// Determine bit-demand of the seperate bands to adjust RES/SFI/Q-bitdemand against each other

#define BITS     (CHAR_BIT * sizeof(*InputBuff))      // Bits per InputBuff-Word
#define INC      InputCnt = (InputCnt + 1) & IBUFMASK

Ibuf_t           InputBuff [IBUFSIZE /* +128 */ ];  // contains the read-buffer
static Uint32_t  mask      [32 + 1];
size_t           InputCnt;             // current position in the read-buffer
static Ibuf_t    dword;                // BITS Bit-Word for Bitstream-I/O
static Uint      pos;                  // position in the currently decoded BITS-bit-Word
static size_t    LastInputCnt = 0;
static Uint      Wraps        = 0;


/*
 *  Initialize all tables and variables
 */

void
Bitstream_init ( void )
{
    Int       i;
    Uint32_t  val;

    InputCnt     = (size_t)-1;
    pos          = BITS;
    dword        = 0;     // Values are initialized in a way that during the next read of at least 1 bit, the first DWORD is collected automatically
    LastInputCnt = 0;
    Wraps        = 0;

    for ( val = 0, i = 0; i <= 32; i++, val += val+1 )
        mask [i] = val;
}


/*
 *  Skip a given number of bits, only forward-skips possible
 */

void
Bitstream_skip ( Uint  bits )
{
    pos     += bits;
    InputCnt = (InputCnt + pos/BITS) & IBUFMASK;
    dword    = InputBuff [InputCnt];
    pos     %= BITS;
}


/*
 *  Read a fixed number of bits from the bitstream. Guaranteed 0...16 bits can be read,
 *  with accesses aligned to 16 bits, it can be up to 32 bits.
 */

Uint32_t
Bitstream_read ( Int bits )
{
    Uint32_t  ret;

    ENTER(78);

    ret = dword;
    if ( (pos += bits) < BITS ) {
        ret >>= BITS - pos;
    }
    else {
        pos  -= BITS;
        INC; ReadLE32 ( dword,  InputBuff+ InputCnt );
        if ( pos > 0 ) {
            ret <<= pos;
            ret  |= dword >> (BITS - pos);
        }
    }
    ret &= mask [bits];

    LEAVE(78);
    REP (printf ( "read(%2u) = %u\n", bits, ret ));
    return ret;
}

/*
 *  Fast form for Bitstream_read(1)
 */

static Uint
Bitstream_read1 ( void )
{
    Uint  ret;

    ENTER(93);

    ret = (Uint)( dword >> ( BITS - ++pos) ) & 1;
    if ( pos >= BITS ) {
        INC; ReadLE32 ( dword,  InputBuff+ InputCnt );
        pos  -= BITS;
    }

    LEAVE(93);
    REP (printf ( "read( 1) = %u\n", ret ));
    return ret;
}


/*
 *  Read of n bits (Restrictions see Bitstream_read), without acknowledging them
 */

Uint32_t
Bitstream_preview ( Int bits )
{
    Uint      new_pos = pos + bits;
    Uint32_t  ret     = dword;
    Uint32_t  tmp;

    if ( new_pos < BITS ) {
        ret >>= BITS - new_pos;
    }
    else if ( new_pos > BITS ) {
        ret <<= new_pos - BITS;
        ReadLE32 ( tmp, & InputBuff [(InputCnt+1) & IBUFMASK] );
        ret  |= tmp >> (2*BITS - new_pos);
    }
    return ret /* & mask[bits] */;
}


/*
 *  Decode Huffman-code, which can be a maximum of 14 bits long.
 *  Decoding simply scans the table.
 */

static Int
Huffman_Decode ( const Huffman_t* Table )
{
    Uint32_t  code;
    Uint32_t  tmp;

    ENTER(79);

    code = dword << pos;
    if ( pos > BITS - 14 ) {
        ReadLE32 ( tmp, & InputBuff [(InputCnt+1) & IBUFMASK] );
        code |= tmp >> (BITS - pos);
    }

    while ( code < Table->Code )
        Table++;

    // Set Bitstream-position without dummy-read
    if ( (pos += Table->Length) >= BITS ) {
        pos   -= BITS;
        INC; ReadLE32 ( dword,  InputBuff+ InputCnt );
    }

    LEAVE(79);
    REP (printf ( "decode() = %d\n", Table->Value ));
    return Table->Value;
}


/*
 *  Decode Huffman-code, which can be a maximum of 14 bits long.
 *  Decoding works with rough positioning via a helper table (tab,unused_bits),
 *  after which it continues to scan until the value is reached.
 */

static Int
Huffman_Decode_faster ( const Huffman_t* Table, const Uint8_t* tab, Int unused_bits )
{
    Uint32_t  code;
    Uint32_t  tmp;

    ENTER(93);

    code = dword << pos;
    if ( pos > BITS - 14 ) {
        ReadLE32 ( tmp, & InputBuff [(InputCnt+1) & IBUFMASK] );
        code |= tmp >> (BITS - pos);
    }

    Table += tab [(size_t)(code >> unused_bits) ];

    while ( code < Table->Code )
        Table++;

    // Set Bitstream-position without dummy-read
    if ( (pos += Table->Length) >= BITS ) {
        pos   -= BITS;
        INC; ReadLE32 ( dword,  InputBuff+ InputCnt );
    }

    LEAVE(93);
    REP (printf ( "decode() = %d\n", Table->Value ));
    return Table->Value;
}

#define HUFFMAN_DECODE_FASTER(a,b,c)  Huffman_Decode_faster ( (a), (b), 32-(c) )


/*
 *  Decode Huffman-code, which can be a maximum of 16 bits long.
 *  Decoding works with a table lookup, therefore only usable for "short" codes,
 *  otherwise we would need huge tables.
 */

static Int
Huffman_Decode_fastest ( const Huffman_t* Table, const Uint8_t* tab, Int unused_bits )
{
    Uint32_t  code;
    Uint32_t  tmp;

    ENTER(91);

    code = dword << pos;
    // is the following line optimal?
    if ( pos > unused_bits + BITS - 32 ) {
        ReadLE32 ( tmp, & InputBuff [(InputCnt+1) & IBUFMASK] );
        code |= tmp >> (BITS - pos);
    }

    Table += tab [ (size_t) (code >> unused_bits) ];

    // Set Bitstream-position without dummy-read
    if ( (pos += Table->Length) >= BITS ) {
        pos   -= BITS;
        INC; ReadLE32 ( dword,  InputBuff+ InputCnt );
    }

    LEAVE(91);
    REP (printf ( "decode() = %d\n", Table->Value ));
    return Table->Value;
}

#define HUFFMAN_DECODE_FASTEST(a,b,c)  Huffman_Decode_fastest ( (a), (b), 32-(c) )


/*
 * Decode huffmann-coded SCFI-Bundle (SV 4...6)
 * Not optimized.
 */

static Uint
SCFIBundle_read ( const Huffman_t* Table ) // is always called with Arg SCFI_Bundle
{
    Uint32_t   code;

    ENTER(81);

    code = dword << pos;
    if (pos > BITS-6)
        code |= InputBuff [(InputCnt+1) & IBUFMASK] >> (BITS-pos);

    while ( code < Table->Code )
        Table++;

    // Set Bitstream-position without dummy-read
    if ( (pos += Table->Length) >= BITS ) {
        pos   -= BITS;
        dword  = InputBuff [InputCnt = (InputCnt+1) & IBUFMASK];
    }

    LEAVE(81);
    return (Uint)Table->Value;
}


Ulong
BitsRead ( void )
{
    if (LastInputCnt > InputCnt) Wraps++;
    LastInputCnt = InputCnt;

    return ((Ulong)Wraps*IBUFSIZE + InputCnt) * BITS + pos;
}

#if  DUMPSELECT > 0
# include "dump.c"
Ulong              __x[8];
# define BITPOS(x)  __x[x] = BitsRead ()
#else
# define BITPOS(x)
#endif

/*
 *  Own function or macro, so that one can centrally modify it
 */

#define Decode_DSCF()   HUFFMAN_DECODE_FASTEST ( HuffDSCF, LUTDSCF, 6 )

/*
 *  Higher resolutions (8 upwards) aren't huffman-coded anymore, number of bits that will then be read directly
 *  Bits per sample for selected resolution, only for the higher resolutions without huffman-coding
 */

#define RES_BIT(x)      ((x)-1)

/******************************************************************************************/
/****************************************** SV 6 ******************************************/
/******************************************************************************************/
void
Read_Bitstream_SV6 ( void )
{
    Int                Band;
    Uint               k;
    const Huffman_t*   Table;
    const Huffman_t*   xL;
    const Huffman_t*   xR;
    Int                Max_Used_Band = 0;

    ENTER(6);

    /********* Read resolution and LR/MS for all Subbands and define last Subband *********************/

    BITPOS(0);
    for ( Band = 0; Band <= Max_Band; Band++ ) {
        Table = Region [Band];

        Res[Band].L = Q_res[Band][Bitrate <= 128  ?  Huffman_Decode(Table)  :  (Int) Bitstream_read(Q_bit[Band])];
        Res[Band].R = 0;

        // Don't read for IS for bands from MinBand+1 on
        if ( !IS_used  ||  Band < Min_Band ) {
            MS_Band[Band] = 0;
            if (MS_used)
                MS_Band[Band] = Bitstream_read1 ();
            Res[Band].R = Q_res[Band][Bitrate <= 128  ?  Huffman_Decode(Table)  :  (Int) Bitstream_read(Q_bit[Band])];
        }
        // Define last used Subband (following operations are just executed up until this one)
        if ( Res[Band].L  ||  Res[Band].R )
            Max_Used_Band = Band;
    }

    /********* Read used Scalebandfactor-Splitting of the last 36 Samples per Subband and Value-addressing (abs./rel.) */

    BITPOS(1);
    for ( Band = 0; Band <= Max_Used_Band; Band++ ) {
        if ( Res[Band].L )
            SCFI[Band].L = SCFIBundle_read (SCFI_Bundle);
        if ( Res[Band].R  ||  (Res[Band].L  &&  IS_used  &&  Band >= Min_Band) )
            SCFI[Band].R = SCFIBundle_read (SCFI_Bundle);
    }

    /********* Read Scalefactors for all Subbands three times for 12 Samples each **************************/

    BITPOS(2);
    Table = DSCF_Entropie;
    for ( Band = 0; Band <= Max_Used_Band; Band++ ) {
        if ( Res[Band].L ) {
            switch ( SCFI[Band].L ) {
            case 0:                                     // without Differential SCF
                SCF_Index[0][Band].L = (Int) Bitstream_read(6);
                SCF_Index[1][Band].L = (Int) Bitstream_read(6);
                SCF_Index[2][Band].L = (Int) Bitstream_read(6);
                break;
            case 2:
                SCF_Index[0][Band].L = (Int) Bitstream_read(6);
                SCF_Index[1][Band].L =
                SCF_Index[2][Band].L = (Int) Bitstream_read(6);
                break;
            case 4:
                SCF_Index[0][Band].L =
                SCF_Index[1][Band].L = (Int) Bitstream_read(6);
                SCF_Index[2][Band].L = (Int) Bitstream_read(6);
                break;
            case 6:
                SCF_Index[0][Band].L =
                SCF_Index[1][Band].L =
                SCF_Index[2][Band].L = (Int) Bitstream_read(6);
                break;
            case 1:                                     // with Differential SCF
                SCF_Index[0][Band].L = SCF_Index[2][Band].L + Huffman_Decode(Table);
                SCF_Index[1][Band].L = SCF_Index[0][Band].L + Huffman_Decode(Table);
                SCF_Index[2][Band].L = SCF_Index[1][Band].L + Huffman_Decode(Table);
                break;
            default:
                assert (0);
            case 3:
                SCF_Index[0][Band].L = SCF_Index[2][Band].L + Huffman_Decode(Table);
                SCF_Index[1][Band].L =
                SCF_Index[2][Band].L = SCF_Index[0][Band].L + Huffman_Decode(Table);
                break;
            case 5:
                SCF_Index[0][Band].L =
                SCF_Index[1][Band].L = SCF_Index[2][Band].L + Huffman_Decode(Table);
                SCF_Index[2][Band].L = SCF_Index[1][Band].L + Huffman_Decode(Table);
                break;
            case 7:
                SCF_Index[0][Band].L =
                SCF_Index[1][Band].L =
                SCF_Index[2][Band].L = SCF_Index[2][Band].L + Huffman_Decode(Table);
                break;
            }
        }

        if ( Res[Band].R  ||  (Res[Band].L  &&  IS_used  &&  Band >= Min_Band) ) {
            switch ( SCFI[Band].R ) {
            case 0:
                SCF_Index[0][Band].R = (Int) Bitstream_read(6);
                SCF_Index[1][Band].R = (Int) Bitstream_read(6);
                SCF_Index[2][Band].R = (Int) Bitstream_read(6);
                break;
            case 2:
                SCF_Index[0][Band].R = (Int) Bitstream_read(6);
                SCF_Index[1][Band].R =
                SCF_Index[2][Band].R = (Int) Bitstream_read(6);
                break;
            case 4:
                SCF_Index[0][Band].R =
                SCF_Index[1][Band].R = (Int) Bitstream_read(6);
                SCF_Index[2][Band].R = (Int) Bitstream_read(6);
                break;
            case 6:
                SCF_Index[0][Band].R =
                SCF_Index[1][Band].R =
                SCF_Index[2][Band].R = (Int) Bitstream_read(6);
                break;
            case 1:
                SCF_Index[0][Band].R = SCF_Index[2][Band].R + Huffman_Decode(Table);
                SCF_Index[1][Band].R = SCF_Index[0][Band].R + Huffman_Decode(Table);
                SCF_Index[2][Band].R = SCF_Index[1][Band].R + Huffman_Decode(Table);
                break;
            default:
                assert (0);
            case 3:
                SCF_Index[0][Band].R = SCF_Index[2][Band].R + Huffman_Decode(Table);
                SCF_Index[1][Band].R =
                SCF_Index[2][Band].R = SCF_Index[0][Band].R + Huffman_Decode(Table);
                break;
            case 5:
                SCF_Index[0][Band].R =
                SCF_Index[1][Band].R = SCF_Index[2][Band].R + Huffman_Decode(Table);
                SCF_Index[2][Band].R = SCF_Index[1][Band].R + Huffman_Decode(Table);
                break;
            case 7:
                SCF_Index[0][Band].R =
                SCF_Index[1][Band].R =
                SCF_Index[2][Band].R = SCF_Index[2][Band].R + Huffman_Decode(Table);
                break;
            }
        }
    }

    /********* Read the quantized Samples per Subband (without Offsets, i.e. values lie zero-symmetric) */

    BITPOS(3);
    for ( Band = 0; Band <= Max_Used_Band; Band++ ) {
        xL = Entropie [Res[Band].L];
        xR = Entropie [Res[Band].R];

        if ( xL != NULL  ||  xR != NULL )
            for (k=0; k<36; k++) {
                if ( xL != NULL )
                    Q[Band].L[k] = Huffman_Decode (xL);
                if ( xR != NULL )
                    Q[Band].R[k] = Huffman_Decode (xR);
            }

        if ( Res[Band].L >= 8  ||  Res[Band].R >= 8 )
            for (k=0; k<36; k++) {
                if ( Res[Band].L >= 8 )
                    Q[Band].L[k] = (Int) Bitstream_read (RES_BIT(Res[Band].L)) - Dc[Res[Band].L];
                if ( Res[Band].R >= 8 )
                    Q[Band].R[k] = (Int) Bitstream_read (RES_BIT(Res[Band].R)) - Dc[Res[Band].R];
            }
    }

    BITPOS(4);
#if  DUMPSELECT > 0
    Dump ( Max_Used_Band, MS_Band, Res, SCF_Index, Q, 6, __x );
#endif

    LEAVE(6);
    return;
}


static Schar  tab30 [3*3*3] = { -1, 0,+1,-1, 0,+1,-1, 0,+1,-1, 0,+1,-1, 0,+1,-1, 0,+1,-1, 0,+1,-1, 0,+1,-1, 0,+1 };
static Schar  tab31 [3*3*3] = { -1,-1,-1, 0, 0, 0,+1,+1,+1,-1,-1,-1, 0, 0, 0,+1,+1,+1,-1,-1,-1, 0, 0, 0,+1,+1,+1 };
static Schar  tab32 [3*3*3] = { -1,-1,-1,-1,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0, 0, 0, 0, 0,+1,+1,+1,+1,+1,+1,+1,+1,+1 };
static Schar  tab50 [5*5  ] = { -2,-1, 0,+1,+2,-2,-1, 0,+1,+2,-2,-1, 0,+1,+2,-2,-1, 0,+1,+2,-2,-1, 0,+1,+2 };
static Schar  tab51 [5*5  ] = { -2,-2,-2,-2,-2,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0,+1,+1,+1,+1,+1,+2,+2,+2,+2,+2 };
#ifdef USE_SV8
static Schar  tab70 [7*7  ] = { -3,-2,-1, 0,+1,+2,+3,-3,-2,-1, 0,+1,+2,+3,-3,-2,-1, 0,+1,+2,+3,-3,-2,-1, 0,+1,+2,+3,-3,-2,-1, 0,+1,+2,+3,-3,-2,-1, 0,+1,+2,+3,-3,-2,-1, 0,+1,+2,+3 };
static Schar  tab71 [7*7  ] = { -3,-3,-3,-3,-3,-3,-3,-2,-2,-2,-2,-2,-2,-2,-1,-1,-1,-1,-1,-1,-1, 0, 0, 0, 0, 0, 0, 0,+1,+1,+1,+1,+1,+1,+1,+2,+2,+2,+2,+2,+2,+2,+3,+3,+3,+3,+3,+3,+3 };
#endif
// 229 Bytes

#if 0
static void
CalculateTNS ( Float* TNS, const CPair_t *Res, const Quant_t* Q, int Band )     // For a non-shaped output the vector should be all 65536 / 5 = 13170.2
{
    int    i;
    int    j;
    Float  Sum;

    for ( i = 0; i < 36; i++ )
        TNS [i] = 3.;

    for ( j = (Band+1)/2; j-- > 0; ) {
        do {
            Q--;
            if (Band-- == 0) goto cont;
        } while ( (--Res) -> L <= 0 );
        for ( i = 0; i < 36; i++ )
            TNS [i] += Q->L [i] * Q->L [i];
    }
cont:

    for ( j = 0; j < 3; j++ ) {
        Sum = 0;
        for ( i = 0; i < 12; i++ )
            Sum += TNS [12*j + i];
        Sum = sqrt (12. / Sum) * 13170.2;
        for ( i = 0; i < 12; i++ )
            TNS [12*j + i] = sqrt (TNS [12*j + i]) * Sum;
    }
}
#endif

/******************************************************************************************/
/****************************************** SV 7 ******************************************/
/******************************************************************************************/

void
Read_Bitstream_SV7 ( void )
{
//  Float               TNS [2] [36];
    Int                 Band;
    Uint                k;
    Int*                p;
    const Huffman_t*    Table;
    Int                 diff;
    Uint                idx;
    Uint32_t            tmp;
    Int                 Max_Used_Band = -1;

    ENTER(7);

    /********* Read resolution and LR/MS for Subband 0 *******************************************************/

    BITPOS(0);
    Res[0].L = (Int) Bitstream_read(4);
    Res[0].R = 0;
    if ( !IS_used  ||  Min_Band > 0 ) {
        Res[0].R = (Int) Bitstream_read(4);
        MS_Band[0] = 0;
        if ( MS_used  &&  (Res[0].L  ||  Res[0].R) )
            MS_Band[0] = Bitstream_read1 ();
    }
    if ( Res[0].L  ||  Res[0].R )
        Max_Used_Band = 0;

    /********* Read resolution and LR/MS for following subbands and determine last Subband *****************/

    Table = HuffHdr;
    for ( Band = 1; Band <= Max_Band; Band++ ) {

        diff = Huffman_Decode (Table);
        Res[Band].L = diff != 4  ?  Res[Band-1].L + diff  :  (Int) Bitstream_read(4);
        Res[Band].R = 0;

        // Don't read for IS for bands from MinBand+1 on
        if ( !IS_used  ||  Min_Band > Band ) {
            diff = Huffman_Decode (Table);
            Res[Band].R = diff != 4  ?  Res[Band-1].R + diff  :  (Int) Bitstream_read(4);
            MS_Band[Band] = 0;
            if ( MS_used  &&  (Res[Band].L  ||  Res[Band].R) )
                switch ( Res[Band].R ) {
                case -3: MS_Band[Band] = Bitstream_read (2) << 2;
                         break;
                case -2: MS_Band[Band] = Bitstream_read (4) << 0;
                         break;
                default: MS_Band[Band] = Bitstream_read1 ();
                         break;
                }
        }
        // Define last used Subband (following operations are just executed up until this one)
        if ( Res[Band].L  ||  Res[Band].R )
            Max_Used_Band = Band;
    }

    /********* Read used Scalebandfactor-Splitting of the last 36 Samples per Subband ************************/

    BITPOS(1);
    Table = HuffSCFI;
    for ( Band = 0; Band <= Max_Used_Band; Band++ ) {
        if ( Res[Band].L > 0  ||  Res[Band].L == -1 )
            SCFI[Band].L = Huffman_Decode (Table);
        if ( (Res[Band].R > 0  ||  Res[Band].R == -1)  ||  (Res[Band].L  &&  IS_used  &&  Band >= Min_Band) )
            SCFI[Band].R = Huffman_Decode (Table);
    }

    /********* Read Scalefaktors for all Subbands three times for 12 Samples each **************************/

    BITPOS(2);
    Table = HuffDSCF;
    for ( Band = 0; Band <= Max_Used_Band; Band++ ) {

        if ( Res[Band].L > 0  ||  Res[Band].L == -1 ) {

            switch ( SCFI[Band].L ) {
            case 0:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].L = diff!=8  ?  SCF_Index[2][Band].L + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[1][Band].L = diff!=8  ?  SCF_Index[0][Band].L + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[2][Band].L = diff!=8  ?  SCF_Index[1][Band].L + diff  :  (Int) Bitstream_read(6);
                break;
            case 1:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].L = diff!=8  ?  SCF_Index[2][Band].L + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[1][Band].L =
                SCF_Index[2][Band].L = diff!=8  ?  SCF_Index[0][Band].L + diff  :  (Int) Bitstream_read(6);
                break;
            case 2:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].L =
                SCF_Index[1][Band].L = diff!=8  ?  SCF_Index[2][Band].L + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[2][Band].L = diff!=8  ?  SCF_Index[1][Band].L + diff  :  (Int) Bitstream_read(6);
                break;
            default:
                assert (0);
            case 3:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].L =
                SCF_Index[1][Band].L =
                SCF_Index[2][Band].L = diff!=8  ?  SCF_Index[2][Band].L + diff  :  (Int) Bitstream_read(6);
                break;
            }
        }

        if ( ( Res[Band].R > 0  ||  Res[Band].R == -1 )  ||  ( Res[Band].L  &&  IS_used  &&  Band >= Min_Band ) ) {

            switch ( SCFI[Band].R ) {
            case 0:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].R = diff!=8  ?  SCF_Index[2][Band].R + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[1][Band].R = diff!=8  ?  SCF_Index[0][Band].R + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[2][Band].R = diff!=8  ?  SCF_Index[1][Band].R + diff  :  (Int) Bitstream_read(6);
                break;
            case 1:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].R = diff!=8  ?  SCF_Index[2][Band].R + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[1][Band].R =
                SCF_Index[2][Band].R = diff!=8  ?  SCF_Index[0][Band].R + diff  :  (Int) Bitstream_read(6);
                break;
            case 2:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].R =
                SCF_Index[1][Band].R = diff!=8  ?  SCF_Index[2][Band].R + diff  :  (Int) Bitstream_read(6);
                diff = Decode_DSCF ();
                SCF_Index[2][Band].R = diff!=8  ?  SCF_Index[1][Band].R + diff  :  (Int) Bitstream_read(6);
                break;
            default:
                assert (0);
            case 3:
                diff = Decode_DSCF ();
                SCF_Index[0][Band].R =
                SCF_Index[1][Band].R =
                SCF_Index[2][Band].R = diff!=8  ?  SCF_Index[2][Band].R + diff  :  (Int) Bitstream_read(6);
                break;
            }
        }
    }

    /********* Read the quantized Samples per Subband (without Offsets, i.e. values lie zero-symmetric) */

    BITPOS(3);
    for ( Band = 0; Band <= Max_Used_Band; Band++ ) {

        p = Q[Band].L;
        switch ( Res[Band].L ) {
        case  -2: case  -3:
            for (k=0; k<36; k++)
                *p++ = 0;
            break;
        case -1:
#if 0
            if ( Res[Band-1].L != -1 )
                CalculateTNS ( TNS[0], (CPair_t *)&Res[Band].L, (Quant_t*)p, Band );
            tmp = random_int ();
            for (k=0; k<36/2; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2)) * TNS [0][k];
            tmp = random_int ();
            for (k=36/2; k<36; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2)) * TNS [0][k];
#elif 0
            tmp = random_int ();
            for (k=0; k<36/2; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2));
            tmp = random_int ();
            for (k=36/2; k<36; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2));
#else
            for (k=0; k<36; k++ ) {
                tmp  = random_int ();
                *p++ = ((tmp >> 24) & 0xFF) + ((tmp >> 16) & 0xFF) + ((tmp >>  8) & 0xFF) + ((tmp >>  0) & 0xFF) - 510;
            }
#endif
            break;
        case 0:
            // Subband samples are not used in this case, see Requant Routines
            break;
        case 1:
            if ( Bitstream_read1 () )
                for (k=0; k<36/3; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ1[1], LUT1_1,  9 );
                    *p++ = tab30[idx];
                    *p++ = tab31[idx];
                    *p++ = tab32[idx];
                }
            else
                for (k=0; k<36/3; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ1[0], LUT1_0,  6 );
                    *p++ = tab30[idx];
                    *p++ = tab31[idx];
                    *p++ = tab32[idx];
                }
            break;
        case 2:
            if ( Bitstream_read1 () )
                for (k=0; k<36/2; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ2[1], LUT2_1, 10 );
                    *p++ = tab50[idx];
                    *p++ = tab51[idx];
                }
            else
                for (k=0; k<36/2; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ2[0], LUT2_0,  7 );
                    *p++ = tab50[idx];
                    *p++ = tab51[idx];
                }
            break;
        case 3:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ3[1], LUT3_1,  5 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ3[0], LUT3_0,  4 );
            break;
        case 4:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ4[1], LUT4_1,  5 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ4[0], LUT4_0,  4 );
            break;
        case 5:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ5[1], LUT5_1,  8 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ5[0], LUT5_0,  6 );
            break;
        case 6:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTER  ( HuffQ6[1], LUT6_1,  7 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ6[0], LUT6_0,  7 );
            break;
        case 7:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTER  ( HuffQ7[1], LUT7_1,  8 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ7[0], LUT7_0,  8 );
            break;
#if 0
            Table = HuffQ [Bitstream_read1 ()] [Res[Band].R];
            for (k=0; k<36; k++)
                *p++ = Huffman_Decode (Table);
            break;
#endif
        case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
            tmp = Dc[Res[Band].L];
            for (k=0; k<36; k++)
                *p++ = (Int) Bitstream_read (RES_BIT(Res[Band].L)) - tmp;
            break;
        default:
            return;
        }

        p = Q[Band].R;
        switch ( Res[Band].R ) {
        case  -2: case  -3:
            for (k=0; k<36; k++)
                *p++ = 0;
            break;
        case -1:
#if 0
            if ( Res[Band-1].R != -1 )
                CalculateTNS ( TNS[1], (CPair_t *)&Res[Band].R, (Quant_t*)p, Band );
            tmp = random_int ();
            for (k=0; k<36/2; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2)) * TNS [1][k];
            tmp = random_int ();
            for (k=36/2; k<36; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2)) * TNS [1][k];
#elif 0
            tmp = random_int ();
            for (k=0; k<36/2; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2));
            tmp = random_int ();
            for (k=36/2; k<36; k++, tmp >>= 1)
                *p++ = (int)(1 - (tmp & 2));
#else
            for (k=0; k<36; k++ ) {
                tmp  = random_int ();
                *p++ = ((tmp >> 24) & 0xFF) + ((tmp >> 16) & 0xFF) + ((tmp >>  8) & 0xFF) + ((tmp >>  0) & 0xFF) - 510;
            }
#endif
            break;
        case 0:
            // Subband samples are not used in this case, see Requant Routines
            break;
        case 1:
            if ( Bitstream_read1 () )
                for (k=0; k<36/3; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ1[1], LUT1_1,  9 );
                    *p++ = tab30[idx];
                    *p++ = tab31[idx];
                    *p++ = tab32[idx];
                }
            else
                for (k=0; k<36/3; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ1[0], LUT1_0,  6 );
                    *p++ = tab30[idx];
                    *p++ = tab31[idx];
                    *p++ = tab32[idx];
                }
            break;
        case 2:
            if ( Bitstream_read1 () )
                for (k=0; k<36/2; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ2[1], LUT2_1, 10 );
                    *p++ = tab50[idx];
                    *p++ = tab51[idx];
                }
            else
                for (k=0; k<36/2; k++) {
                    idx  = HUFFMAN_DECODE_FASTEST ( HuffQ2[0], LUT2_0,  7 );
                    *p++ = tab50[idx];
                    *p++ = tab51[idx];
                }
            break;
        case 3:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ3[1], LUT3_1,  5 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ3[0], LUT3_0,  4 );
            break;
        case 4:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ4[1], LUT4_1,  5 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ4[0], LUT4_0,  4 );
            break;
        case 5:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ5[1], LUT5_1,  8 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ5[0], LUT5_0,  6 );
            break;
        case 6:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTER  ( HuffQ6[1], LUT6_1,  7 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ6[0], LUT6_0,  7 );
            break;
        case 7:
            if ( Bitstream_read1 () )
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTER  ( HuffQ7[1], LUT7_1,  8 );
            else
                for (k=0; k<36; k++)
                    *p++ = HUFFMAN_DECODE_FASTEST ( HuffQ7[0], LUT7_0,  8 );
            break;
#if 0
            Table = HuffQ [Bitstream_read1 ()] [Res[Band].R];
            for (k=0; k<36; k++)
                *p++ = Huffman_Decode (Table);
            break;
#endif
        case 8: case 9: case 10: case 11: case 12: case 13: case 14: case 15: case 16: case 17:
            tmp = Dc[Res[Band].R];
            for (k=0; k<36; k++)
                *p++ = (Int) Bitstream_read (RES_BIT(Res[Band].R)) - tmp;
            break;
        default:
            return;
        }
    }

    BITPOS(4);

#if  DUMPSELECT > 0
    Dump ( Max_Used_Band, MS_Band, Res, SCF_Index, Q, 7, __x );
#endif

    LEAVE(7);
    return;
}


/******************************************************************************************/
/****************************************** SV 8 ******************************************/
/******************************************************************************************/

#ifdef USE_SV8
void
Read_Bitstream_SV8 ( void )
{
    Int                Band;
    Int                k;
    Int                Channel;
    Int                idx;
    Int                Max_Channel;
    Int                Max_Used_Band;
    CPair_t*           Res_p;
    CPairArray*        SCF_p;
    Int*               p;
    const Huffman_t*   Table;

    ENTER(8);

    memset ( Res    , 0, sizeof(Res)     );
    memset ( MS_Band, 0, sizeof(MS_Band) );
    memset ( Q      , 0, sizeof(Q)       );

    /********************************* Header *****************************/
    Max_Used_Band = (Int) Bitstream_read(6) - 1; // maximum non-zero band
    if ( (Int)Max_Used_Band < 0 )
        return;

    Max_Channel   = (Int) Bitstream_read (3);    // number of channels
    MS_used       = Bitstream_read1 ();          // M/S-coding

    /******************************* Res ******************************/
    for ( Channel = 0; Channel < Max_Channel; Channel++ ) {
        Res_p = Channel  ?  (CPair_t*)&(Res[0].R)  :  (CPair_t*)&(Res[0].L);

        for ( Band = 0; Band <= Max_Used_Band; Band++ )
            Res_p[Band].L = (Int) Bitstream_read (4);
    }

    /******************************* MS *******************************/
    if ( MS_used ) {
        for ( Channel = 0; Channel < Max_Channel-1; Channel++) {
            for ( Band = 0; Band <= Max_Used_Band; Band++ )
                MS_Band[Band] = Bitstream_read1 ();
        }
    }

    /**************************** SCF/DSCF ****************************/
    for ( Channel = 0; Channel < Max_Channel; Channel++ ) {
        if ( Channel == 0 ) {
            SCF_p = (CPairArray*) &(SCF_Index [0][0].L);
            Res_p = (CPair_t*) &(Res [0].L);
        } else {
            SCF_p = (CPairArray*) &(SCF_Index [0][0].R);
            Res_p = (CPair_t*) &(Res [0].R);
        }

        for ( Band = 0; Band <= Max_Used_Band; Band++ ) {
            if ( Res_p[Band].L ) {
                SCF_p[0][Band].L = (Int) Bitstream_read (7);
                SCF_p[1][Band].L = (Int) Bitstream_read (7);
                SCF_p[2][Band].L = (Int) Bitstream_read (7);
            }
        }
    }

    /***************************** Samples ****************************/
    for ( Channel = 0; Channel < Max_Channel; Channel++ ) {
        if ( Channel == 0 ) {
            p     = Q[0].L;
            Res_p = (CPair_t*)&(Res[0].L);
        } else {
            p     = Q[0].R;
            Res_p = (CPair_t*)&(Res[0].R);
        }

        for ( Band = 0; Band <= Max_Used_Band; Band++, p+=36 ) {
            REP (printf ("Channel %u, Band %2u (%2u)\n", Channel, Band, Res_p[Band].L ));
            switch ( Res_p[Band].L ) {
            case 0:
                p += 36;
                break;
            case 1:
                Table = HuffN [(Int) Bitstream_read(1)] [1];
                for (k=0; k<36/3; k++) {
                    idx = Huffman_Decode (Table);
                    *p++ = tab30[idx];
                    *p++ = tab31[idx];
                    *p++ = tab32[idx];
                }
                break;
            case 2:
                Table = HuffN [(Int) Bitstream_read(1)] [2];
                for (k=0; k<36/2; k++) {
                    idx = Huffman_Decode (Table);
                    *p++ = tab50[idx];
                    *p++ = tab51[idx];
                }
                break;
            case 3:
                Table = HuffN [(Int) Bitstream_read(1)] [3];
                for (k=0; k<36/2; k++) {
                    idx = Huffman_Decode (Table);
                    *p++ = tab70[idx];
                    *p++ = tab71[idx];
                }
                break;
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
                Table = HuffN [(Int) Bitstream_read(1)][Res_p[Band].L];
                for (k=0; k<36; k++)
                    *p++ = Huffman_Decode (Table);
                break;
            default:
                for (k=0; k<36; k++)
                    *p++ = (Int) Bitstream_read (RES_BIT(Res_p[Band].L)) - Dc[Res_p[Band].L];
                break;
            }
        }
    }

# if  DUMPSELECT > 0
    Dump ( Max_Used_Band, MS_Band, Res, SCF_Index, Q, 8, __x );
#endif

    LEAVE(8);
    return;
}
#endif

/* end of decode.c */
