/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "mppdec.h"

Int8_t      Q_bit [32];         // Number of bits to save the resolution (SV6)
Int8_t      Q_res [32] [16];    // Index -> resolution (SV6)
Float       __SCF [6 + 128];    // tabulated scalefactors with safety margin
Float       __Cc  [1 + 18];     // Requantization-coefficients
const Uint  __Dc  [1 + 18] = {  // Requantization-Offset
      2,
      0,     1,     2,     3,     4,     7,    15,    31,    63,
    127,   255,   511,  1023,  2047,  4095,  8191, 16383, 32767
};
Uint        Bitrate;
Int         Min_Band;
Int         Max_Band;


// Initialize Min_Band, Max_Band; Input is function-parameter and bitrate

static void
Set_BandLimits ( Int Max_Band_desired, Bool_t used_IS )
{
    if ( Max_Band_desired > 0 ) {   // Bandwidth as chosen by user
        Max_Band = Max_Band_desired;
    } else {                        // Default-Bandwidth
        if      ( Bitrate > 384 ) Max_Band = 31;        // 22.05 kHz
        else if ( Bitrate > 160 ) Max_Band = 29;        // 20.67 kHz
        else if ( Bitrate >  64 ) Max_Band = 26;        // 18.60 kHz
        else if ( Bitrate >   0 ) Max_Band = 20;        // 14.47 kHz
        else                      Max_Band = 31;        // 22.05 kHz
    }

    if ( used_IS ) {
        if      ( Bitrate > 384 ) assert (0);
        else if ( Bitrate > 160 ) Min_Band = 16;        // 11.02 kHz
        else if ( Bitrate > 112 ) Min_Band = 12;        //  8.27 kHz
        else if ( Bitrate > 64  ) Min_Band =  8;        //  5.51 kHz
        else                      Min_Band =  4;        //  2.76 kHz

        if ( Min_Band >= Max_Band )
            Min_Band = Max_Band /* + 1 ????? */;
    } else {
        Min_Band = Max_Band + 1;
    }
}


// Initialize Q_bit and Q_res

static void
Set_QuantizeMode_SV4_6 ( void )
{
    Int  Band;
    Int  i;

    for ( Band = 0; Band < 11; Band++ ) {
        Q_bit [Band] = 4;
        for ( i = 0; i < (1<<4)-1; i++ )
            Q_res [Band] [i] = (Uint8_t)i;
        Q_res [Band] [(1<<4)-1] = 17;
    }
    for ( Band = 11; Band < 23; Band++ ) {
        Q_bit [Band] = 3;
        for ( i = 0; i < (1<<3)-1; i++ )
            Q_res [Band] [i] = (Uint8_t)i;
        Q_res [Band] [(1<<3)-1] = 17;
    }
    for ( Band = 23; Band < 32; Band++ ) {
        Q_bit [Band] = 2;
        for ( i = 0; i < (1<<2)-1; i++ )
            Q_res [Band] [i] = (Uint8_t)i;
        Q_res [Band] [(1<<2)-1] = 17;
    }
}


// Initialize table SCF

static void
Calc_ScaleFactors ( Ldouble start, Ldouble mult )
{
    size_t  i;

    for ( i = 0; i < sizeof(__SCF)/sizeof(*__SCF); i++ ) {
        __SCF [i] = (Float) start;
        start    *= mult;
    }
}


// Initialize Cc, needs table Dc

static void
Calc_RequantTab_SV4_7 ( void )
{
    size_t  i;

    __Cc [0] = 111.28596247532739441973f;                                       // 16384 / 255 * sqrt(3)
    for ( i = 1; i < sizeof(__Cc)/sizeof(*__Cc); i++ )
        __Cc [i] = (Float) (32768. / (__Dc [i] + 0.5));
}


static void
Calc_RequantTab_SV8 ( void )
{
    size_t  i;

    for ( i = 0; i < sizeof(__Cc)/sizeof(*__Cc); i++ )
        __Cc [i] = (Float) 1.;
}


#define C1  1.20050805774840750476L
#define C2  0.83298066476582673961L
#define C3  0.501187233627272285285L
#define C4  1.122018454301963435591L

void
Init_QuantTab ( Int maximum_Band, Bool_t used_IS, Double amplification, Uint StreamVersion )
{
    // Initializations are independent from each other, order is arbitrary
    Set_BandLimits ( maximum_Band, used_IS );
    Set_QuantizeMode_SV4_6 ();

    if ( (StreamVersion & 15) < 8 ) {
        Calc_RequantTab_SV4_7 ();
        // Covers the range +7.936...-98.4127 dB, where scf[n]/scf[n-1] = 1.20050805774840750476
        Calc_ScaleFactors ( amplification * C1/(C2*C2*C2*C2*C2*C2), C2 );
    } else {
        Calc_RequantTab_SV8 ();
        Calc_ScaleFactors ( amplification * C3/(C4*C4*C4*C4*C4*C4), C4 );
    }
}

/* end of requant.c */
