//-----------------------------------------------------------------------------
//
//	Musepack Demuxer
//
//	Author : Igor Janos
//
//-----------------------------------------------------------------------------
#include "stdafx.h"



//-----------------------------------------------------------------------------
//
//	CMPCFile class
//
//-----------------------------------------------------------------------------

CMPCFile::CMPCFile() :
	duration_10mhz(0),
	reader(NULL),
	packet(NULL),
	seek_table(NULL),
	seek_table_position(0)
{
	key = 0;
	packet_max_size = 512 * 1024;			// well. hard to say. but I think this should be enough
	packet_size = 0;
	packet = (uint8*)malloc(packet_max_size);

	ASSERT(packet);
}

CMPCFile::~CMPCFile()
{
	if (seek_table) {
		free(seek_table);
		seek_table = NULL;
	}
	if (packet) {
		free(packet);
		packet = NULL;
	}
}

// I/O for MPC file
int CMPCFile::Open(CMPCReader *reader)
{
	HRESULT		hr;
	int			ret;
	bool		done;

	// keep a local copy of the reader
	this->reader = reader;


	// According to stream specification the first 4 bytes should be 'MPCK'
	uint32	magick;
	reader->Seek(0);
	ret = reader->GetMagick(magick);			if (ret < 0) return ret;
	if (magick != 0x4d50434b) {
		// not a Musepack file
		return -1;
	}

	// means no seek table
	seek_table_position = 0;
	seek_table_size = 0;

	done = false;

	// now loop through a few packets
	do {
		ret = ReadNextPacket();
		if (ret < 0) return -1;			// end of file reached before initialization is done ?

		switch (key) {
		case MPC_KEY('S','H'):	ret = ReadStreamHeader(); break;
		case MPC_KEY('R','G'):  ret = ReadReplaygain(); break;
		case MPC_KEY('S','O'):	ret = ReadSeekOffset(); break;

		// audio packet - initialization is over
		case MPC_KEY('A','P'):	done = true; break;
		}

		// if parsing failed we quit
		if (ret < 0) return ret;
	} while (!done);

	// if there was a seek table, load it
	if (seek_table_position != 0) {
		ret = ReadSeekTable(); 
		if (ret < 0) return ret;
	}

	return 0;
}

int CMPCFile::ReadNextPacket()
{
	uint16	key_val;
	int64	size_val;
	int32	size_len;
	int		ret;
	int64	avail;

	reader->GetPosition(&packet_pos, &avail);
	ret = reader->GetKey(key_val);					if (ret < 0) return ret;
	ret = reader->GetSizeElm(size_val, size_len);	if (ret < 0) return ret;

	// if the key is not valid, quit
	if (!reader->KeyValid(key_val)) return -1;
	key = key_val;

	// size_val is the total size including key, variable size field and payload
	packet_size = size_val - 2 - size_len;
	ASSERT(packet_size <= packet_max_size);

	ret = reader->Read(packet, packet_size);		if (ret < 0) return ret;
	return 0;
}

// parsing packets

int CMPCFile::ReadReplaygain()
{
	Bitstream	b(packet);
	b.NeedBits();

	int version = b.UGetBits(8);
	if (version != 1) return 0;		// unsupported RG version. not critical to us...

	int16	val;
	b.NeedBits();	val = b.SGetBits(16);	gain_title_db = val / 256.0;
	b.NeedBits();	val = b.SGetBits(16);	gain_title_peak_db = val / 256.0;
	b.NeedBits();	val = b.SGetBits(16);	gain_album_db = val / 256.0;
	b.NeedBits();	val = b.SGetBits(16);	gain_album_peak_db = val / 256.0;

	return 0;
}

int CMPCFile::ReadSeekOffset()
{
	Bitstream	b(packet);

	seek_table_position = b.GetMpcSize();
	seek_table_position += packet_pos;

	// success 
	return 0;
}

int CMPCFile::ReadSeekTable()
{
	reader->Seek(seek_table_position);
	int ret;

	ret = ReadNextPacket();
	if (ret < 0) return ret;

	Bitstream	b(packet);

	// calculate size as seen in mpc_demux.c
	int64 tmp = b.GetMpcSize();	b.NeedBits();
	seek_table_size = tmp;
	int	seek_pwr = block_pwr + b.UGetBits(4);
	tmp = 2+total_samples / (1152 << seek_pwr);
	if (tmp < seek_table_size) tmp = seek_table_size;

	// alloc memory for seek table
	seek_table = (uint32*)malloc(tmp * sizeof(uint32));

	uint32 *table = seek_table;

	tmp = b.GetMpcSize();
	table[0] = (uint32)(tmp + header_position) * 8;
	if (seek_table_size == 1) return 0;

	tmp = b.GetMpcSize();
	table[1] = (uint32)(tmp + header_position) * 8;
	for (int i=2; i<seek_table_size; i++) {
		int code = b.Get_Golomb(12);
		if (code&1) code = -(code&(-1 << 1));
		code <<= 2;
		table[i] = code + 2*table[i-1] - table[i-2];
	}

	return 0;
}

int CMPCFile::ReadStreamHeader()
{
	Bitstream	b(packet);
	b.NeedBits();

	// let's do some reading
	uint32		crc;
	crc = b.UGetBits(16);	b.NeedBits();
	crc <<= 16;
	crc |= b.UGetBits(16);

	// TODO: CRC checking.
	b.NeedBits();
	stream_version = b.UGetBits(8);

	switch (stream_version) {
	case 7:
		{
			//-----------------------------------------------------------------
			//	Stream Version 7
			//-----------------------------------------------------------------

			// since we don't do any actual decoding here, we can only look 
			// for the fields of our interest

			// TODO: later. now we want SV8

		}
		break;
	case 8:
		{
			//-----------------------------------------------------------------
			//	Stream Version 8
			//-----------------------------------------------------------------
			total_samples = b.GetMpcSize();
			int64	silence = b.GetMpcSize();

			b.NeedBits();
			uint8	freq = b.UGetBits(3);
			switch (freq) {
			case 0:	sample_rate = 44100; break;
			case 1:	sample_rate = 48000; break;
			case 2:	sample_rate = 37800; break;
			case 3: sample_rate = 32000; break;
			default:
				// invalid value
				return -1;				
			}

			// let's calculate duration
			duration_10mhz = (total_samples * 10000000) / sample_rate;

			b.DumpBits(5);					// bands
			channels = b.UGetBits(4)+1;		// channels

			b.NeedBits();
			b.DumpBits(1);					// mid side
			block_pwr = b.UGetBits(3) * 2;
			audio_block_frames = 1 << (block_pwr);

			// store absolute position of stream header
			header_position = packet_pos;
		}
		break;
	default:
		// stream version not supported
		return -2;
	}

	// everything is okay
	return 0;
}


