//-----------------------------------------------------------------------------
//
//	Musepack Demuxer
//
//	Author : Igor Janos
//
//-----------------------------------------------------------------------------
#include "stdafx.h"



//-----------------------------------------------------------------------------
//
//	CMPCFile class
//
//-----------------------------------------------------------------------------

CMPCFile::CMPCFile() :
	duration_10mhz(0),
	reader(NULL),
	seek_table(NULL),
	seek_table_position(0)
{
	extradata_max_size = 16*1024;			// way too much..
	extradata_size = 0;
	extradata = (uint8*)malloc(extradata_max_size);
}

CMPCFile::~CMPCFile()
{
	if (seek_table) { free(seek_table); seek_table = NULL; }
	if (extradata) { free(extradata); extradata = NULL; }
}

// I/O for MPC file
int CMPCFile::Open(CMPCReader *reader)
{
	HRESULT		hr;
	int			ret;
	bool		done;

	// keep a local copy of the reader
	this->reader = reader;

	// According to stream specification the first 4 bytes should be 'MPCK'
	uint32	magick;
	reader->Seek(0);
	ret = reader->GetMagick(magick);			if (ret < 0) return ret;
	if (magick != 0x4d50434b) {
		// not a Musepack file
		return -1;
	}

	// means no seek table
	seek_table_position = 0;
	seek_table_size = 0;

	done = false;

	// init extradata
	extradata[0] = 'M';
	extradata[1] = 'P';
	extradata[2] = 'C';
	extradata[3] = 'K';
	extradata_size = 4;

	// now loop through a few packets
	CMPCPacket		packet;
	do {
		ret = packet.Load(reader);
		if (ret < 0) return -1;			// end of file reached before initialization is done ?

		switch (packet.key) {
		case MPC_KEY('S','H'):	
			ret = ReadStreamHeader(&packet); 
			StoreExtraDataPacket(&packet);
			break;
		case MPC_KEY('R','G'):  
			ret = ReadReplaygain(&packet); 
			StoreExtraDataPacket(&packet);
			break;
		case MPC_KEY('S','O'):	
			ret = ReadSeekOffset(&packet); 
			break;
		case MPC_KEY('E','I'):
			StoreExtraDataPacket(&packet);
			break;

		// audio packet - initialization is over
		case MPC_KEY('A','P'):	done = true; break;
		}

		// if parsing failed we quit
		if (ret < 0) return ret;
	} while (!done);

	// insert a dummy AP packet
	uint8 *out = extradata + extradata_size;
	out[0] = 'A';
	out[1] = 'P';
	out[2] = 0x03;
	extradata_size += 3;

	// if there was a seek table, load it
	if (seek_table_position != 0) {
		int64 first_ap_pos, avail;
		reader->GetPosition(&first_ap_pos, &avail);

		reader->Seek(seek_table_position);
		ret = packet.Load(reader);
		ret = ReadSeekTable(&packet); 
		if (ret < 0) return ret;

		// seek back to first AP
		reader->Seek(first_ap_pos);
	}

	// we're at start
	current_sample = 0;
	return 0;
}

void CMPCFile::StoreExtraDataPacket(CMPCPacket *packet)
{
	// we add current packet to extradata
	uint8	*out = extradata + extradata_size;
	memcpy(out, packet->packet, packet->packet_size);
	extradata_size += packet->packet_size;
}

// parsing packets
int CMPCFile::ReadReplaygain(CMPCPacket *packet)
{
	Bitstream	b(packet->payload);
	b.NeedBits();

	int version = b.UGetBits(8);
	if (version != 1) return 0;		// unsupported RG version. not critical to us...

	int16	val;
	b.NeedBits();	val = b.SGetBits(16);	gain_title_db = val / 256.0;
	b.NeedBits();	val = b.SGetBits(16);	gain_title_peak_db = val / 256.0;
	b.NeedBits();	val = b.SGetBits(16);	gain_album_db = val / 256.0;
	b.NeedBits();	val = b.SGetBits(16);	gain_album_peak_db = val / 256.0;

	return 0;
}

int CMPCFile::ReadSeekOffset(CMPCPacket *packet)
{
	Bitstream	b(packet->payload);

	seek_table_position = b.GetMpcSize();
	seek_table_position += packet->file_position;

	// success 
	return 0;
}

int CMPCFile::ReadSeekTable(CMPCPacket *packet)
{
	Bitstream	b(packet->payload);

	// calculate size as seen in mpc_demux.c
	int64 tmp = b.GetMpcSize();	b.NeedBits();
	seek_table_size = tmp;
	seek_pwr = block_pwr + b.UGetBits(4);
	tmp = 2+total_samples / (1152 << seek_pwr);
	if (tmp < seek_table_size) tmp = seek_table_size;

	// alloc memory for seek table
	seek_table = (uint32*)malloc(tmp * sizeof(uint32));

	uint32 *table = seek_table;

	tmp = b.GetMpcSize();
	table[0] = (uint32)(tmp + header_position) * 8;
	if (seek_table_size == 1) return 0;

	tmp = b.GetMpcSize();
	table[1] = (uint32)(tmp + header_position) * 8;
	for (int i=2; i<seek_table_size; i++) {
		int code = b.Get_Golomb(12);
		if (code&1) code = -(code&(-1 << 1));
		code <<= 2;
		table[i] = code + 2*table[i-1] - table[i-2];
	}

	return 0;
}

int CMPCFile::ReadStreamHeader(CMPCPacket *packet)
{
	Bitstream	b(packet->payload);
	b.NeedBits();

	// let's do some reading
	uint32		crc;
	crc = b.UGetBits(16);	b.NeedBits();
	crc <<= 16;
	crc |= b.UGetBits(16);

	// TODO: CRC checking.
	b.NeedBits();
	stream_version = b.UGetBits(8);

	switch (stream_version) {
	case 7:
		{
			//-----------------------------------------------------------------
			//	Stream Version 7
			//-----------------------------------------------------------------

			// since we don't do any actual decoding here, we can only look 
			// for the fields of our interest

			// TODO: later. now we want SV8

		}
		break;
	case 8:
		{
			//-----------------------------------------------------------------
			//	Stream Version 8
			//-----------------------------------------------------------------
			total_samples = b.GetMpcSize();
			int64	silence = b.GetMpcSize();

			b.NeedBits();
			uint8	freq = b.UGetBits(3);
			switch (freq) {
			case 0:	sample_rate = 44100; break;
			case 1:	sample_rate = 48000; break;
			case 2:	sample_rate = 37800; break;
			case 3: sample_rate = 32000; break;
			default:
				// invalid value
				return -1;				
			}

			// let's calculate duration
			duration_10mhz = (total_samples * 10000000) / sample_rate;

			b.DumpBits(5);					// bands
			channels = b.UGetBits(4)+1;		// channels

			b.NeedBits();
			b.DumpBits(1);					// mid side
			block_pwr = b.UGetBits(3) * 2;
			audio_block_frames = 1 << (block_pwr);

			// store absolute position of stream header
			header_position = packet->file_position -4;
		}
		break;
	default:
		// stream version not supported
		return -2;
	}

	// everything is okay
	return 0;
}

// parsing out packets
int CMPCFile::ReadAudioPacket(CMPCPacket *packet, int64 *cur_sample)
{
	// we just load packets until we face the SE packet. Then we return -1
	int ret;

	do {
		ret = packet->Load(reader);
		if (ret < 0) {
			return ret;
		}

		switch (packet->key) {
		case MPC_KEY('A','P'):	
			{
				// keep track of samples...
				if (cur_sample) *cur_sample = current_sample;
				current_sample += (1152*audio_block_frames);

				return 0;			// we got one
			}
		case MPC_KEY('S','E'):
			{
				return -2;
			}
			break;
		}

		// skip other packets
	} while (1);

	// unexpected...
	return 0;
}

int CMPCFile::Seek(int64 seek_sample)
{
	/*
		This will be a little tricky. I would like to support also 
		less precise seeking with files that don't have seeking table. 
		But if there is one, we	would use that one.

		Original implementation also deals with introductory silence.
		But since I don't know of any dshow filters capable of
		gapless playback we would ignore this. Of course it can
		be implemented later.

		It is absolutely okay to do AP-level seeking. The caller
		knows the exact time it wants to seek to and we will provide
		the nearest AP. If the requested time is in the middle of AP,
		the (current_time - requested_time) expression would be negative
		and decoder will skip samples until positive time values are encountered.
		This way we can do sample-precise seeking.
	*/

	// cannot seek
	if (seek_table == NULL || seek_table_size == 0) return -1;

	int packet_num = seek_sample / (1152*audio_block_frames);
	int i = (packet_num >> (seek_pwr - block_pwr));
	if (i >= seek_table_size) {
		i = seek_table_size-1;
	}

	// seek to position
	uint32 pos = seek_table[i] >> 3;
	reader->Seek(pos);

	// shift back
	i = i << (seek_pwr - block_pwr);
	current_sample = i*audio_block_frames;

	return 0;
}


//-----------------------------------------------------------------------------
//
//	CMPCPacket
//
//-----------------------------------------------------------------------------

CMPCPacket::CMPCPacket() :
	file_position(0),
	packet(NULL),
	payload(NULL),
	packet_size(0),
	payload_size(0),
	key(0)
{
}

CMPCPacket::~CMPCPacket()
{
	// just let it go
	Release();
}

void CMPCPacket::Release()
{
	if (packet) { free(packet); packet = NULL; }
	payload = NULL;
	packet_size = 0;
	payload_size = 0;
	key = 0;
	file_position = 0;
}

int CMPCPacket::Load(CMPCReader *reader)
{
	uint16	key_val;
	int64	size_val, avail;
	int32	size_len;
	int		ret;
	Release();
	reader->GetPosition(&file_position, &avail);

	// end of stream
	if (file_position >= avail) {
		return -2;
	}

	ret = reader->GetKey(key_val);					
	if (ret < 0) {
		return ret;
	}
	ret = reader->GetSizeElm(size_val, size_len);	
	if (ret < 0) {
		return ret;
	}

	// if the key is not valid, quit
	if (!reader->KeyValid(key_val)) {
		return -1;
	}
	key = key_val;

	// now load the packet
	packet_size  = size_val;
	payload_size = size_val - 2 - size_len;
	packet = (uint8*)malloc(packet_size);
	payload = packet + 2 + size_len;				// pointer to packet payload

	// roll back the bytes
	reader->Seek(file_position);
	ret = reader->Read(packet, packet_size);		
	if (ret < 0) {
		return ret;
	}

	return 0;
}


