/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <assert.h>
#include <time.h>

#include <mpcdec/mpcdec.h>
#include <libwaveformat.h>

static void
usage(const char *exename)
{
    printf
        ("Usage: %s <infile.mpc> [<outfile.wav>]\nIf <outfile.wav> is not specified, decoder will run in benchmark mode.\n",
         exename);
}

int
main(int argc, char **argv)
{
    mpc_reader reader; mpc_streaminfo si; mpc_decoder decoder; wavwriter

    if (argc != 2 && argc != 3) {
        if (argc > 0)
            usage(argv[0]);
        return 0;
    }


    FILE *input = fopen(argv[1], "rb");
    FILE *output = 0;
    if (input == 0) {
        usage(argv[0]);
        printf("Error opening input file: \"%s\"\n", argv[1]);
        return 1;
    }

    if (argc == 3) {
        output = fopen(argv[2], "wb");
        if (output == 0) {
            fclose(input);
            usage(argv[0]);
            printf("Error opening output file: \"%s\"\n", argv[2]);
            return 1;
        }
    }

    /* initialize our reader_data tag the reader will carry around with it */
    reader_data data;
    data.file = input;
    data.seekable = true;
    fseek(data.file, 0, SEEK_END);
    data.size = ftell(data.file);
    fseek(data.file, 0, SEEK_SET);

    /* set up an mpc_reader linked to our function implementations */
    mpc_decoder decoder;
    mpc_reader reader;
    reader.read = read_impl;
    reader.seek = seek_impl;
    reader.tell = tell_impl;
    reader.get_size = get_size_impl;
    reader.canseek = canseek_impl;
    reader.data = &data;

    /* read file's streaminfo data */
    mpc_streaminfo info;
    mpc_streaminfo_init(&info);
    if (mpc_streaminfo_read(&info, &reader) != ERROR_CODE_OK) {
        printf("Not a valid musepack file: \"%s\"\n", argv[1]);
        return 1;
    }

    /* instantiate a decoder with our file reader */
    mpc_decoder_setup(&decoder, &reader);
    if (!mpc_decoder_initialize(&decoder, &info)) {
        printf("Error initializing decoder.\n", argv[1]);
        return 1;
    }

    /* decode the file */
    printf("Decoding from:\n%s\nTo:\n%s\n", argv[1],
           output ? argv[2] : "N/A");
    WavWriter *wavwriter =
        output ? new WavWriter(output, 2, 16, info.sample_freq) : 0;
    MPC_SAMPLE_FORMAT sample_buffer[MPC_DECODER_BUFFER_LENGTH];
    clock_t begin, end;
    begin = clock();
    unsigned total_samples = 0;
    mpc_bool_t successful = FALSE;
    for (;;) {
        unsigned status = mpc_decoder_decode(&decoder, sample_buffer, 0, 0);
        if (status == (unsigned)(-1)) {
            //decode error
            printf("Error decoding file.\n");
            break;
        }
        else if (status == 0)   //EOF
        {
            successful = true;
            break;
        }
        else                    //status>0
        {
            total_samples += status;
            if (wavwriter) {
                if (!wavwriter->
                    WriteSamples(sample_buffer, status * /* stereo */ 2)) {
                    printf("Write error.\n");
                    break;
                }
            }
        }
    }

    end = clock();

    if (wavwriter) {
        delete wavwriter;
    }

    if (successful) {
        printf("\nFinished.\nTotal samples decoded: %u.\n", total_samples);
        unsigned ms = (end - begin) * 1000 / CLOCKS_PER_SEC;
        unsigned ratio =
            (unsigned)((double)total_samples / (double)info.sample_freq /
                       ((double)ms / 1000.0) * 100.0);
        printf("Time: %u ms (%u.%02ux).\n", ms, ratio / 100, ratio % 100);
    }

    return 0;
}
