/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/// \file streaminfo.c
/// Implementation of streaminfo reading functions.

#include <mpcdec/streaminfo.h>
#include <stdio.h>
#include "internal.h"

static const char na[] = "n.a.";
static const char *versionNames[] = {
    na, "'Unstable/Experimental'", na, na, na, "'quality 0'", "'quality 1'", 
    "'Telephone'", "'Thumb'", "'Radio'", "'Standard'", "'Extreme'", "'Insane'",
    "'BrainDead'", "'quality 9'", "'quality 10'"
};
static const mpc_int32_t samplefreqs[4] = { 44100, 48000, 37800, 32000 };

static const char *
mpc_get_version_string(mpc_uint32_t profile) // profile is 0...15, where 7...13 is used
{
    return profile >= sizeof versionNames / sizeof *versionNames ? na : versionNames[profile];
}

void
mpc_streaminfo_init(mpc_streaminfo* si)
{
    memset(si, 0, sizeof *si);
}

/// Reads streaminfo from SV7 header. 
static mpc_status
streaminfo_read_header_sv7(mpc_streaminfo* si, mpc_uint32_t HeaderData[8])
{
    mpc_uint16_t Estimatedpeak_title = 0;

    if (si->stream_version > 0x71) {
        //        Update (si->stream_version);
        return MPC_STATUS_OK;
    }

    si->bitrate            = 0;
    si->frames             = HeaderData[1];
    si->is                 = 0;
    si->ms                 = (HeaderData[2] >> 30) & 0x0001;
    si->max_band           = (HeaderData[2] >> 24) & 0x003F;
    si->block_size         = 1;
    si->profile            = (HeaderData[2] << 8) >> 28;
    si->profile_name       = mpc_get_version_string(si->profile);
    si->sample_freq        = samplefreqs[(HeaderData[2] >> 16) & 0x0003];
    Estimatedpeak_title    = (mpc_uint16_t) (HeaderData[2] & 0xFFFF);   // read the ReplayGain data
    si->gain_title         = (mpc_uint16_t) ((HeaderData[3] >> 16) & 0xFFFF);
    si->peak_title         = (mpc_uint16_t) (HeaderData[3] & 0xFFFF);
    si->gain_album         = (mpc_uint16_t) ((HeaderData[4] >> 16) & 0xFFFF);
    si->peak_album         = (mpc_uint16_t) (HeaderData[4] & 0xFFFF);
    si->is_true_gapless    = (HeaderData[5] >> 31) & 0x0001; // true gapless: used?
    si->last_frame_samples = (HeaderData[5] >> 20) & 0x07FF; // true gapless: valid samples for last frame
    si->fast_seek          = (HeaderData[5] >> 19) & 0x0001; // fast seeking
    si->encoder_version    = (HeaderData[6] >> 24) & 0x00FF;
    si->channels           = 2;

    if (si->encoder_version == 0) {
        sprintf(si->encoder, "Buschmann 1.7.0...9, Klemm 0.90...1.05");
    }
    else {
        switch (si->encoder_version % 10) {
        case 0:
            sprintf(si->encoder, "Release %u.%u", si->encoder_version / 100,
                    si->encoder_version / 10 % 10);
            break;
        case 2: case 4: case 6: case 8:
            sprintf(si->encoder, "Beta %u.%02u", si->encoder_version / 100,
                    si->encoder_version % 100);
            break;
        default:
            sprintf(si->encoder, "--Alpha-- %u.%02u",
                    si->encoder_version / 100, si->encoder_version % 100);
            break;
        }
    }

    return MPC_STATUS_OK;
}

// reads file header and tags
mpc_status
mpc_streaminfo_read(mpc_streaminfo * si, mpc_reader *p_reader)
{
    mpc_uint32_t headerData[8]; int err;

    // get header position
    err = si->header_position = mpc_skip_id3v2(p_reader);
    if(err < 0) return MPC_STATUS_FILE;

    // seek to first byte of mpc data
    err = p_reader->seek(p_reader, si->header_position);
    if(!err) return MPC_STATUS_FILE;
    err = p_reader->read(p_reader, headerData, 8 * 4);
    if(err != 8 * 4) return MPC_STATUS_FILE;
    err = p_reader->seek(p_reader, si->header_position + 6 * 4);
    if(!err) return MPC_STATUS_FILE;

    err = p_reader->get_size(p_reader);
    if(err < 0) return MPC_STATUS_FILE;
    si->tag_offset = si->total_file_length = err;

    err = memcmp(headerData, "MP+", 3);
    if(!err) return MPC_STATUS_INVALIDSV;

#ifndef MPC_LITTLE_ENDIAN
    {
        mpc_uint32_t ptr;
        for (ptr = 0; ptr < 8; ptr++)
            headerData[ptr] = mpc_swap32(headerData[ptr]);
    }
#endif
     si->stream_version = headerData[0] >> 24;

     if ((si->stream_version & 15) >= 8)
        return MPC_STATUS_INVALIDSV;
     else if ((si->stream_version & 15) == 7)
     {
        err = streaminfo_read_header_sv7(si, headerData);
        if(err < 0) return err;
     }

    // estimation, exact value needs too much time
    si->pcm_samples     = 1152 * si->frames - 576;
    si->average_bitrate = 0;
    if (si->pcm_samples > 0)
        si->average_bitrate = (si->tag_offset  - si->header_position) * 8.0 
                            *  si->sample_freq / si->pcm_samples;

    return MPC_STATUS_OK;
}

double
mpc_streaminfo_get_length(mpc_streaminfo * si)
{
    return (double) mpc_streaminfo_get_length_samples(si) / si->sample_freq;
}

mpc_int64_t
mpc_streaminfo_get_length_samples(mpc_streaminfo * si)
{
    mpc_int64_t samples = (mpc_int64_t) si->frames * MPC_FRAME_LENGTH;
    if (si->is_true_gapless)
        samples -= (MPC_FRAME_LENGTH - si->last_frame_samples);
    else
        samples -= MPC_DECODER_SYNTH_DELAY;
    return samples;
}
