/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
/// \file mpcdec.h
/// Top level include file for libmpcdec.
#ifndef _MPCDEC_H_
#define _MPCDEC_H_
#ifdef WIN32
#pragma once
#endif

#include "streaminfo.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct mpc_decoder_t mpc_decoder;

/// Sets up decoder library.
/// Call this first when preparing to decode an mpc stream.
/// \param p_reader reader that will supply raw data to the decoder
void mpc_decoder_setup(mpc_decoder *p_dec, mpc_reader *p_reader);

/// Initializes mpc decoder with the supplied stream info parameters.
/// \param si streaminfo structure indicating format of source stream
/// \return TRUE if decoder was initalized successfully, FALSE otherwise    
mpc_status mpc_init_decoder(mpc_decoder **p_dec, mpc_streaminfo *si);

/// Releases input mpc decoder 
void mpc_exit_decoder(mpc_decoder *p_dec);

/// Call this next after calling mpc_decoder_setup.
/// \param si streaminfo structure indicating format of source stream
/// \param fast_seeking boolean 0 = use fast seeking if safe, 1 = force fast seeking
void mpc_decoder_set_seeking(mpc_decoder *p_dec, mpc_streaminfo *si, mpc_bool_t fast_seeking);

void mpc_decoder_set_streaminfo(mpc_decoder *p_dec, mpc_streaminfo *si);

/// Sets decoder sample scaling factor.  All decoded samples will be multiplied
/// by this factor.
/// \param scale_factor multiplicative scaling factor
void mpc_decoder_scale_output(mpc_decoder *p_dec, double scale_factor);

/// Actually reads data from previously initialized stream.  Call
/// this iteratively to decode the mpc stream.
/// \param buffer destination buffer for decoded samples
/// \param vbr_update_acc \todo document me
/// \param vbr_update_bits \todo document me
/// \return -1 if an error is encountered
/// \return 0 if the stream has been completely decoded successfully and there are no more samples
/// \return > 0 to indicate the number of bytes that were actually read from the stream.
mpc_uint32_t mpc_decoder_decode(
    mpc_decoder       *p_dec,
    MPC_SAMPLE_FORMAT *buffer, 
    mpc_uint32_t      *vbr_update_acc, 
    mpc_uint32_t      *vbr_update_bits);

mpc_uint32_t mpc_decoder_decode_frame(
    mpc_decoder       *p_dec,
    mpc_uint32_t      *in_buffer,
    mpc_uint32_t      in_len,
    MPC_SAMPLE_FORMAT *out_buffer);

/// Seeks to the specified sample in the source stream.
mpc_bool_t mpc_decoder_seek_sample(mpc_decoder *p_dec, mpc_int64_t destsample);

/// Seeks to specified position in seconds in the source stream.
mpc_bool_t mpc_decoder_seek_seconds(mpc_decoder *p_dec, double seconds);

#ifdef __cplusplus
}
#endif
#endif
