/*
  Copyright (c) 2005, The Musepack Development Team
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are
  met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

  * Redistributions in binary form must reproduce the above
  copyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials provided
  with the distribution.

  * Neither the name of the The Musepack Development Team nor the
  names of its contributors may be used to endorse or promote
  products derived from this software without specific prior
  written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <stdio.h>
#include <assert.h>
#include <time.h>
#include <mpcdec/mpcdec.h>

#ifdef WIN32
#include <crtdbg.h>
#endif

static void
usage(const char *exename)
{
    printf("Usage: %s <infile.mpc>\n", exename);
}

int
main(int argc, char **argv)
{
    mpc_reader reader; mpc_streaminfo si; mpc_decoder* decoder; mpc_status err;
    MPC_SAMPLE_FORMAT sample_buffer[MPC_DECODER_BUFFER_LENGTH];
    clock_t begin, end, sum; int total_samples;

    if(argc != 2)
    {
        usage(argv[0]);
        return 0;
    }

    err = mpc_reader_init_stdio(&reader, argv[1]);
    if(err < 0) return !MPC_STATUS_OK;

    err = mpc_streaminfo_init(&si, &reader);
    if(err < 0) return !MPC_STATUS_OK;

    err = mpc_decoder_init(&decoder, &reader, &si);
    if(err < 0) return !MPC_STATUS_OK;

    sum = total_samples = 0;
    while(TRUE)
    {
        mpc_uint32_t samples;
        begin = clock();
        samples = mpc_decoder_decode(decoder, sample_buffer, 0, 0);
        end = clock();
        if(samples <= 0) break;
        if(!samples) break;
        total_samples += samples;
        sum           += end - begin;
    }

    printf("%u samples ", total_samples);
    total_samples = (mpc_uint32_t) ((double) total_samples * CLOCKS_PER_SEC * 100 / (si.sample_freq * sum));
    printf("decoded in %u ms (%u.%02ux)\n",
           sum * 1000 / CLOCKS_PER_SEC,
           total_samples / 100,
           total_samples % 100
           );

    mpc_decoder_exit(decoder);
    mpc_reader_exit_stdio(&reader);

#ifdef WIN32
    assert(_CrtCheckMemory());
    _CrtDumpMemoryLeaks();
#endif
    return 0;
}
