/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>

#include "libmpcenc.h"
#include "stdio.h"


/*
 *  Change_Endian32() changes the endianess of a 32-bit memory block in-place
 *  by swapping the byte order. This is a little bit tricky, but a well
 *  known method which is much much faster, especially on modern CPUs, than
 *  byte picking, because it avoids memory aliasing. Note that this method
 *  is poison for old 16-bit compilers!
 */

#if ENDIAN == HAVE_BIG_ENDIAN

static void
Change_Endian32 ( unsigned int* dst, mpc_size_t words32bit )
{
    for ( ; words32bit--; dst++ ) {
# if  INT_MAX >= 2147483647L
        unsigned int  tmp = *dst;
        tmp  = ((tmp << 0x10) & 0xFFFF0000) | ((tmp >> 0x10) & 0x0000FFFF);
        tmp  = ((tmp << 0x08) & 0xFF00FF00) | ((tmp >> 0x08) & 0x00FF00FF);
        *dst = tmp;
# else
        char  tmp;
        tmp             = ((char*)dst)[0];
        ((char*)dst)[0] = ((char*)dst)[3];
        ((char*)dst)[3] = tmp;
        tmp             = ((char*)dst)[1];
        ((char*)dst)[1] = ((char*)dst)[2];
        ((char*)dst)[2] = tmp;
# endif
    }
    return;
}

#endif /* ENDIAN == HAVE_BIG_ENDIAN */


void
FlushBitstream ( FILE* fp, const mpc_uint32_t* buffer, mpc_size_t words32bit )
{
    mpc_size_t           WrittenDwords = 0;
    const mpc_uint32_t*  p             = buffer;
#if ENDIAN == HAVE_BIG_ENDIAN
    mpc_size_t           CC            = words32bit;
#endif

#if ENDIAN == HAVE_BIG_ENDIAN
    Change_Endian32 ( (mpc_uint32_t*)buffer, CC );
#endif

    // Write e->Buffer
    do {
        WrittenDwords = fwrite ( p, sizeof(*buffer), words32bit, fp );
        if ( WrittenDwords == 0 ) {
			// FIXME : move stderr_printf to common
//             stderr_printf ( "\b\n WARNING: Disk full?, retry after 10 sec ...\a" );
			sprintf(stderr, "\b\n WARNING: Disk full?, retry after 10 sec ...\a");
            sleep (10);
        }
        if ( WrittenDwords > 0 ) {
            p          += WrittenDwords;
            words32bit -= WrittenDwords;
        }
    } while ( words32bit != 0 );

#if ENDIAN == HAVE_BIG_ENDIAN
    Change_Endian32 ( (mpc_uint32_t*)buffer, CC );
#endif
}


void
UpdateHeader ( FILE* fp, mpc_uint32_t Frames, mpc_uint_t ValidSamples )
{
    mpc_uint8_t  buff [4];

    // Write framecount to header
    if ( fseek ( fp, 4L, SEEK_SET ) < 0 )
        return;

    buff [0] = (mpc_uint8_t)(Frames >>  0);
    buff [1] = (mpc_uint8_t)(Frames >>  8);
    buff [2] = (mpc_uint8_t)(Frames >> 16);
    buff [3] = (mpc_uint8_t)(Frames >> 24);

    fwrite ( buff, 1, 4, fp );

    // Write ValidSamples to header
    if ( fseek ( fp, 22L, SEEK_SET ) < 0 )
        return;
    fread ( buff, 1, 2, fp );
    if ( ferror(fp) )
        return;
    if ( fseek ( fp, 22L, SEEK_SET ) < 0 )
        return;

    ValidSamples <<= 4;
    ValidSamples  |= 0x800F & (((mpc_uint_t) buff[1] << 8) | buff[0]);
    buff [0] = (mpc_uint8_t)(ValidSamples >>  0);
    buff [1] = (mpc_uint8_t)(ValidSamples >>  8);

    fwrite ( buff, 1, 2, fp );


    // Set filepointer to end of file (dirty method, should be old position!!)
    fseek ( fp, 0L, SEEK_END );
}


void WriteBits (mpc_encoder_t * e, const mpc_uint32_t input, const unsigned int bits )
{
    e->BufferedBits += bits;
    e->filled       -= bits;

    if      ( e->filled > 0 ) {
        e->dword  |= input << e->filled;
    }
    else if ( e->filled < 0 ) {
        e->Buffer [e->Zaehler++] = e->dword | ( input >> -e->filled );
        e->filled += 32;
        e->dword   = input << e->filled;
    }
    else {
        e->Buffer [e->Zaehler++] = e->dword | input;
        e->filled  = 32;
        e->dword   =  0;
    }
}

// Bits in the original stream have to be 0, maximum X bits allowed to be set in input
// Actual bitstream must have already written ptr[0] and ptr[1]
void WriteBitsAt (mpc_encoder_t * e, const mpc_uint32_t input, const unsigned int bits, BitstreamPos const pos )
{
    mpc_uint32_t*     ptr    = pos.ptr;
    int           filled = pos.bit - bits;

//    fprintf ( stderr, "%5u %2u %08lX %2u\n", input, bits, pos.ptr, pos.bit );

    e->Buffer [e->Zaehler] = e->dword;

    if      ( filled > 0 ) {
        ptr [0] |= input << (  +filled);
    }
    else if ( filled < 0 ) {
        ptr [0] |= input >> (  -filled);
        ptr [1] |= input << (32+filled);
    }
    else {
        ptr [0] |= input;
    }

    e->dword = e->Buffer [e->Zaehler];
}


void GetBitstreamPos (mpc_encoder_t * e, BitstreamPos* const pos )
{
    pos -> ptr = e->Buffer + e->Zaehler;
    pos -> bit = e->filled;
}

/* end of bitstream.c */
