/*
 * Musepack audio compression
 * Copyright (C) 1999-2004 Buschmann/Klemm/Piecha/Wolf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 *  A list of different mixed tools
 *  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *  Read_LittleEndians()
 *      Portable file handling
 *  Requantize_MidSideStereo(), Requantize_IntensityStereo()
 *      Requantisation of quantized samples for synthesis filter
 *  Resort_HuffTable(), Make_HuffTable(), Make_LookupTable()
 *      Generating and sorting Huffman tables, making fast lookup tables
 */

#include <string.h>
#include <errno.h>
#include <stdio.h>

// FIXME : not sure if it's a good idea
#include <fpu_control.h>

#include "libmpcenc.h"

// READ    returns -1 or 0 on error/EOF, otherwise > 0
#define READ(fp,ptr,len)       fread  (ptr, 1, len, fp)

#if defined HAVE_INCOMPLETE_READ  &&  FILEIO != 1

mpc_size_t
complete_read ( int fd, void* dest, mpc_size_t bytes )
{
    mpc_size_t  bytesread = 0;
    mpc_size_t  ret;

    while ( bytes > 0 ) {
        ret = read ( fd, dest, bytes );
        if ( ret == 0  ||  ret == (mpc_size_t)-1 )
            break;
        dest       = (void*)(((char*)dest) + ret);
        bytes     -= ret;
        bytesread += ret;
    }
    return bytesread;
}

#endif


/*
 *  Change_Endian32() changes the endianess of a 32-bit memory block in-place
 *  by swapping the byte order. This is a little bit tricky, but a well
 *  known method which is much much faster than byte picking, especially on modern CPUs,
 *  because it avoids memory aliasing. Note that this method
 *  is poison for old 16-bit compilers!
 */

#if ENDIAN == HAVE_BIG_ENDIAN

static void
Change_Endian32 ( mpc_uint32_t* dst, mpc_size_t words32bit )
{

    for ( ; words32bit--; dst++ ) {
# if  INT_MAX >= 2147483647L
        mpc_uint32_t  tmp = *dst;
        tmp  = ((tmp << 0x10) & 0xFFFF0000) | ((tmp >> 0x10) & 0x0000FFFF);
        tmp  = ((tmp << 0x08) & 0xFF00FF00) | ((tmp >> 0x08) & 0x00FF00FF);
        *dst = tmp;
# else
        mpc_uint8_t  tmp;
        tmp                = ((mpc_uint8_t*)dst)[0];
        ((mpc_uint8_t*)dst)[0] = ((mpc_uint8_t*)dst)[3];
        ((mpc_uint8_t*)dst)[3] = tmp;
        tmp                = ((mpc_uint8_t*)dst)[1];
        ((mpc_uint8_t*)dst)[1] = ((mpc_uint8_t*)dst)[2];
        ((mpc_uint8_t*)dst)[2] = tmp;
# endif
    }
    return;
}

#endif /* ENDIAN == HAVE_BIG_ENDIAN */


/*
 *  Read_LittleEndians() reads little endian 32-bit ints from the stream
 *  'fp'.  Quantities are selected in 32-bit items. On big endian machines
 *  the byte order is changed in-place after reading the data, so all is
 *  okay.
 */

mpc_size_t
Read_LittleEndians ( FILE* fp, mpc_uint32_t* dst, mpc_size_t words32bit )
{
    mpc_size_t  wordsread;
    wordsread = READ ( fp, dst, words32bit * sizeof(*dst) ) / sizeof(*dst);

#if ENDIAN == HAVE_BIG_ENDIAN
    Change_Endian32 ( dst, wordsread );
#endif

    return wordsread;
}

/*
 *  Fills out the items Code and Length (but not Value) of a Huffman table
 *  from a bit packed Huffman table 'src'. Table is not sorted, so this is
 *  the table which is suitable for an encoder. Be careful: To get a table
 *  usable for a decoder you must use Resort_HuffTable() after this
 *  function. It's a little bit dangerous to divide the functionality, maybe
 *  there is a more secure and handy solution to this problem.
 */

void
Make_HuffTable ( Huffman_t* dst, const HuffSrc_t* src, mpc_size_t len )
{
    mpc_size_t  i;

    for ( i = 0; i < len; i++,src++,dst++ ) {
        dst->Code   = src->Code  ;
        dst->Length = src->Length;
    }
}


/*
 *  Generates a Lookup table for quick Huffman decoding. This table must
 *  have a size of a power of 2. Input is the pre-sorted Huffman table,
 *  sorted by Resort_HuffTable() and its length, and the length of the
 *  lookup table. Output is the Lookup table. It can be used for table based
 *  decoding (Huffman_decode_fastest) which fully decodes by means of the
 *  LUT. This is only handy for small huffman codes up to 9...10 bit
 *  maximum length. For longer codes partial lookup is possible with
 *  Huffman_decode_faster() which first estimates possible codes by means
 *  of LUT and then searches the exact code like the tableless version
 *  Huffman_decode().
 */

void
Make_LookupTable ( mpc_uint8_t* LUT, mpc_size_t LUT_len, const Huffman_t* const Table, const mpc_size_t elements )
{
    mpc_size_t    i;
    mpc_size_t    idx  = elements;
    mpc_uint32_t  dval = (mpc_uint32_t)0x80000000L / LUT_len * 2;
    mpc_uint32_t  val  = dval - 1;

    for ( i = 0; i < LUT_len; i++, val += dval ) {
        while ( idx > 0  &&  val >= Table[idx-1].Code )
            idx--;
        *LUT++ = (mpc_uint8_t)idx;
    }

    return;
}


void
Init_FPU ( void )
{
    mpc_uint16_t  cw;

#if   defined __i386__  &&  defined _FPU_GETCW  &&  defined _FPU_SETCW
    _FPU_GETCW ( cw );
    cw  &=  ~0x300;
    _FPU_SETCW ( cw );
#elif defined __i386__  &&  defined  FPU_GETCW  &&  defined  FPU_SETCW
    FPU_GETCW ( cw );
    cw  &=  ~0x300;
    FPU_SETCW ( cw );
#elif defined __MINGW32__
    __asm__ ("fnstcw %0" : "=m" (*&cw));
    cw  &=  ~0x300;
    __asm__ ("fldcw %0" : : "m" (*&cw));
#elif defined(_WIN32) && !defined(_WIN64)
    _asm { fstcw cw };
    cw  &=  ~0x300;
    _asm { fldcw cw };
#else
    ;
#endif
}


